import {Component, OnInit} from '@angular/core';
import {Validators} from '@angular/forms';
import {NgbModal} from '@ng-bootstrap/ng-bootstrap';
import {CrudComponent} from '@app/shared/crud.component';
import {FormModalComponent, FormModalInputOptions} from '@app/shared/form-modal/form-modal.component';
import {Entity} from '@app/shared/models/entity.model';
import {School} from '@app/shared/models/school.model';
import {Router} from '@angular/router';
import {SchoolsDataService} from '@app/schools/schools-data.service';
import {OrganisationsDataService} from "@app/organisations/organisations-data.service";
import {Organisation} from "@app/shared/models/organisation.model";
import { NotificationsService } from 'angular2-notifications';
import { tap } from 'rxjs/operators';

@Component({
  selector: 'app-schools',
  templateUrl: './schools-list.component.html',
  styleUrls: ['./schools-list.component.scss']
})
export class SchoolsListComponent extends CrudComponent<School> implements OnInit {
  organisation: Organisation = null;

  constructor(modal: NgbModal, private schoolsService: SchoolsDataService,
              private organisationsService: OrganisationsDataService,
              private toast: NotificationsService,
              private router: Router) {
    super();
    super.initialize(modal);
  }

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.loadSchools();
  }

  loadSchools() {
    this.schoolsService.getAllForOrganisation(this.organisation.id).subscribe(schools => this.entities = schools);
  }

  rowClicked(event: any) {
    this.router.navigate(["/organisaties", this.organisation.id, "scholen", event.selected[0].id]);
  }

  deleteEntity(entity: School) {
    this.schoolsService.deleteFromOrganisation(this.organisation.id, entity.id).subscribe(() => {
      this.toast.success("School verwijderd.");
      this.loadSchools();
    });
  }

  editPressed(entity: Entity) {
    const school = entity as School;

    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = 'School aanpassen';
    component.confirmLabel = 'Aanpassen';
    component.fields = [
      new FormModalInputOptions('name', 'Naam van de school', 'text', school.name, [Validators.required])
    ];
    component.callback = (result: any) => this.schoolsService.updateInOrganisation(this.organisation.id, {...result, id: entity.id});
  }

  newPressed() {
    const modalRef = this.modalService.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = 'Nieuwe school toevoegen';
    component.confirmLabel = 'Toevoegen';
    component.fields = [
      new FormModalInputOptions('name', 'Naam van de school', 'text', '', [Validators.required]),
      new FormModalInputOptions('prefix', 'Prefix', 'text', '', [Validators.required])
    ];
    component.callback = (result: any) => this.schoolsService.addToOrganisation(this.organisation.id, result).pipe(tap(() => this.loadSchools()));
  }
}
