import { Component, OnInit, ViewChild } from "@angular/core";
import { Member } from "@app/shared/models/member.model";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { SchoolsDataService } from "../schools-data.service";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { Validators } from "@angular/forms";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { NotificationsService } from "angular2-notifications";
import { DeleteModalComponent } from "@app/shared/delete-modal/modal.component";
import { DatatableComponent } from "@swimlane/ngx-datatable";
import { forkJoin } from "rxjs";
import { Title } from "@angular/platform-browser";
import { School } from "@app/shared/models/school.model";
import { Router } from "@angular/router";
import { Organisation } from "@app/shared/models/organisation.model";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";

@Component({
  selector: "app-students",
  templateUrl: "./members.component.html",
  styleUrls: ["./members.component.scss"]
})
export class MembersComponent implements OnInit {
  members: Member[] = [];
  accessLevels: number[] = [5, 6, 7];
  accessLevelsEnabled = [true, true, false];

  organisation: Organisation;
  school: School;

  temp: Member[] = [];
  unassigned: Member[] = [];
  selected: any[] = [];
  @ViewChild(DatatableComponent) table: DatatableComponent;

  constructor(
    private modal: NgbModal,
    private schoolsService: SchoolsDataService,
    private organisationsService: OrganisationsDataService,
    private title: Title,
    private router: Router,
    private toast: NotificationsService
  ) {}

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.school = this.schoolsService.currentSchool;
    this.getMembers();
  }

  getMembers() {
    this.schoolsService
      .getAllMembers(
        this.organisation.id,
        this.school.id,
        this.accessLevels.filter((a, i) => this.accessLevelsEnabled[i])
      )
      .subscribe(members => {
        this.temp = members;
        this.members = members;
      });

    this.title.setTitle(`Scholen > ${this.school.name} > Accounts`);
  }

  newPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Nieuw account toevoegen";
    component.description = "Je kunt meerdere accounts tegelijk toevoegen door ze met komma's te scheiden.";
    component.confirmLabel = "Toevoegen";
    component.fields = [
      new FormModalInputOptions("firstName", "Voornaam", "text", "", [Validators.required]),
      new FormModalInputOptions("insertion", "Tussenvoegsel", "text", ""),
      new FormModalInputOptions("lastName", "Achternaam", "text", "", [Validators.required]),
      new FormModalInputOptions("username", "E-mailadres", "email", "", [Validators.required]),
      new FormModalInputOptions("accessLevelId", "Type", "select", {
        value: "6",
        options: [
          // {
          //   id: "5",
          //   name: "Beheerder"
          // },
          {
            id: "6",
            name: "Begeleider"
          }
        ].map(type => new DropdownOption(type.id, type.name))
      })
    ];
    component.callback = result => {
      const accountsToCreate = result.username.split(",").map((username: string) =>
        this.schoolsService.addMember(this.organisation.id, this.school.id, {
          username: username,
          // @ts-ignore
          accessLevelId: parseInt(result.accessLevelId, 10)
        })
      );

      return forkJoin(accountsToCreate);
    };
    component.onSuccess = () => {
      this.toast.success("Schoolaccount toegevoegd");
      this.getMembers();
    };

    component.onError = err => {
      this.toast.error("Er ging iets mis.", err.error.message);
      this.getMembers();
    };
  }

  rowClicked(event: any) {
    this.router.navigate([
      "/organisaties",
      this.organisation.id,
      "scholen",
      this.school.id,
      "accounts",
      event.selected[0].id
    ]);
  }

  deletePressed(member: Member) {
    const modalRef = this.modal.open(DeleteModalComponent);
    const component = modalRef.componentInstance as DeleteModalComponent;
    component.name = `Account ${member.username}`;
    modalRef.result
      .then(result => {
        this.schoolsService.deleteMember(this.organisation.id, this.school.id, member).subscribe(() => {
          this.toast.success(`Schoolaccount is verwijderd.`);
          this.getMembers();
        });
      })
      .catch(() => {});
  }

  updateFilter(event: any) {
    const val = event.target.value.toLowerCase();

    // filter our data
    const temp = this.temp.filter(d => {
      return (
        JSON.stringify(d)
          .toLowerCase()
          .indexOf(val) !== -1 || !val
      );
    });

    // update the rows
    this.members = temp;
    // Whenever the filter changes, always go back to the first page
    this.table.offset = 0;
  }

  getAdmins() {
    return this.members.filter(m => m.accessLevel === "schoolAdmin");
  }

  getGuides() {
    return this.members.filter(m => m.accessLevel === "schoolGuide");
  }

  getStudents() {
    return this.members.filter(m => m.accessLevel === "schoolStudent");
  }

  updateMembersList() {
    this.getMembers();
  }
}
