import { Contact } from "./../shared/models/contact.model";
import { EventEmitter, Injectable } from "@angular/core";
import { HttpClient } from "@angular/common/http";

import { Observable } from "rxjs";

import { map, tap } from "rxjs/operators";

import { Member } from "@app/shared/models/member.model";
import { School } from "@app/shared/models/school.model";
import { Group } from "@app/shared/models/group.model";
import { Batch } from "@app/shared/models/batch.model";

@Injectable()
export class SchoolsDataService {
  schools: { [schoolId: string]: School[] } = {};
  _currentSchool: School = null;
  schoolChanged = new EventEmitter();

  get currentSchool(): School {
    return this._currentSchool;
  }

  set currentSchool(value: School) {
    this._currentSchool = value;
    this.schoolChanged.emit(value);
  }

  constructor(private http: HttpClient) {}

  getAllForOrganisation(organisationId: string): Observable<School[]> {
    return this.http.get<School[]>(`/organisations/${organisationId}/schools`).pipe(
      map(schools => schools.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))),
      tap(schools => (this.schools[organisationId] = schools))
    );
  }

  getByIdForOrganisation(organisationId: string, key: string | number): Observable<School> {
    return this.http.get<School>(`/organisations/${organisationId}/schools/${key}`);
  }

  addToOrganisation(organisationId: string, entity: School): Observable<School> {
    return this.http.post<School>(`/organisations/${organisationId}/schools`, entity);
  }

  deleteFromOrganisation(organisationId: string, entityId: string): Observable<void> {
    return this.http.delete<void>(`/organisations/${organisationId}/schools/${entityId}`);
  }

  updateInOrganisation(organisationId: string, entity: Partial<School>) {
    return this.http.put<School>(`/organisations/${organisationId}/schools/${entity.id}`, entity);
  }

  getMembers(organisationId: string, schoolId: string): Observable<Member[]> {
    return this.http.get<Member[]>(`/organisations/${organisationId}/schools/${schoolId}/admins`);
  }

  getMember(organisationId: string, schoolId: string, memberId: string): Observable<Member> {
    return this.http.get<Member>(`/organisations/${organisationId}/schools/${schoolId}/members/${memberId}`);
  }

  addMember(organisationId: string, schoolId: string, member: Member) {
    return this.http.post<Member>(`/organisations/${organisationId}/schools/${schoolId}/members`, member);
  }

  updateMember(organisationId: string, schoolId: string, data: any) {
    return this.http.put<Member>(`/organisations/${organisationId}/schools/${schoolId}/members/${data.id}`, data);
  }

  deleteMember(organisationId: string, schoolId: string, member: Member) {
    return this.http.delete<void>(`/organisations/${organisationId}/schools/${schoolId}/members/${member.id}`);
  }

  getContacts(organisationId: string, schoolId: string): Observable<Contact[]> {
    return this.http.get<Contact[]>(`/organisations/${organisationId}/schools/${schoolId}/contacts`);
  }

  addContact(organisationId: string, schoolId: string, contact: Contact) {
    return this.http.post<Contact>(`/organisations/${organisationId}/schools/${schoolId}/contacts`, contact);
  }

  updateContact(organisationId: string, schoolId: string, data: any) {
    return this.http.put<Member>(`/organisations/${organisationId}/schools/${schoolId}/contacts/${data.id}`, data);
  }

  deleteContact(organisationId: string, schoolId: string, contact: Contact) {
    return this.http.delete<void>(`/organisations/${organisationId}/schools/${schoolId}/contacts/${contact.id}`);
  }

  getAllGroupsWithId(organisationId: string, schoolId: string, eventId: string = ""): Observable<Group[]> {
    const url =
      eventId === ""
        ? `/organisations/${organisationId}/schools/${schoolId}/groups`
        : `/organisations/${organisationId}/schools/${schoolId}/groups?eventId=${eventId}`;

    return this.http
      .get<Group[]>(url)
      .pipe(map(groups => groups.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))));
  }

  getGroupWithId(organisationId: string, schoolId: string, groupId: string): Observable<Group> {
    return this.http.get<Group>(`/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}`);
  }

  getAllGroups(organisationId: string, schoolId: string, eventId: string = ""): Observable<Group[]> {
    const url =
      eventId === ""
        ? `/organisations/${organisationId}/schools/${schoolId}/groups`
        : `/organisations/${organisationId}/schools/${schoolId}/groups?eventId=${eventId}`;

    return this.http
      .get<Group[]>(url)
      .pipe(map(groups => groups.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))));
  }

  getAllGroupsForSchool(organisationId: string, schoolId: string, eventId: string = null): Observable<Group[]> {
    return this.http
      .get<Group[]>(
        `/organisations/${organisationId}/schools/${schoolId}/groups${eventId != null ? "?eventId=" + eventId : ""}`
      )
      .pipe(map(groups => groups.sort((a, b) => (a.name.toLowerCase() < b.name.toLowerCase() ? -1 : 1))));
  }

  addGroup(organisationId: string, schoolId: string, name: string, eventId: string): Observable<Group> {
    return this.http.post<Group>(`/organisations/${organisationId}/schools/${schoolId}/groups`, {
      eventId,
      name
    });
  }

  updateGroup(organisationId: string, schoolId: string, entity: Group) {
    return this.http.put(`/organisations/${organisationId}/schools/${schoolId}/groups/${entity.id}`, entity);
  }

  deleteGroup(organisationId: string, schoolId: string, group: Group) {
    return this.http.delete(`/organisations/${organisationId}/schools/${schoolId}/groups/${group.id}`);
  }

  importStudents(organisationId: string, schoolId: string, payload: any): Observable<Member[]> {
    return this.http.post<Member[]>(`/organisations/${organisationId}/schools/${schoolId}/import`, payload);
  }

  getBatches(organisationId: string, schoolId: string): Observable<Batch[]> {
    return this.http.get<Batch[]>(`/organisations/${organisationId}/schools/${schoolId}/import/batches`);
  }

  deleteBatch(organisationId: string, schoolId: string, batchId: string): Observable<void> {
    return this.http.delete<void>(`/organisations/${organisationId}/schools/${schoolId}/import/batches/${batchId}`);
  }

  getAllMembers(
    organisationId: string,
    schoolId: string,
    accessLevels?: number[],
    unassigned?: boolean
  ): Observable<Member[]> {
    return this.http.get<Member[]>(`/organisations/${organisationId}/schools/${schoolId}/members`, {
      params: {
        accessLevels: accessLevels ? JSON.stringify(accessLevels) : undefined,
        unassigned: unassigned ? "true" : undefined
      }
    });
  }

  getAllGroupMembers(organisationId: string, schoolId: string, groupId: string): Observable<Member[]> {
    return this.http.get<Member[]>(`/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}/members`);
  }

  addGroupMember(organisationId: string, schoolId: string, groupId: string, memberId: string[]): Observable<void> {
    return this.http.post<void>(`/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}/members`, {
      userId: memberId
    });
  }

  updateGroupMember(organisationId: string, schoolId: string, groupId: string, member: Member): Observable<Member> {
    return this.http.put<Member>(
      `/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}/members/${member.id}`,
      member
    );
  }

  deleteGroupMember(organisationId: string, schoolId: string, groupId: string, memberId: string): Observable<void> {
    return this.http.delete<void>(
      `/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}/members/${memberId}`
    );
  }

  updateGroupCategories(
    organisationId: string,
    schoolId: string,
    groupId: string,
    categories: number[]
  ): Observable<void> {
    return this.http.put<void>(`/organisations/${organisationId}/schools/${schoolId}/groups/${groupId}`, {
      categories
    });
  }
}
