import { HttpClient, HttpHeaders } from '@angular/common/http';
import {Injectable} from '@angular/core';
import { AuthDataService, LoginResponse } from './auth-data.service';

@Injectable({
  providedIn: "root"
})
export class AuthService {
  constructor(private http: HttpClient, private authDataService: AuthDataService) {
    const permissions = JSON.parse(window.localStorage.getItem("permissions")) ?? null;
    const user = JSON.parse(window.localStorage.getItem("user")) ?? null;
    const authToken = JSON.parse(window.localStorage.getItem("authToken")) ?? null;

    this.authDataService.permissions = permissions;
    this.authDataService.user = user;
    this.authDataService.authToken = authToken;

    this.authDataService.triggerLogout.subscribe(() => this.logout());
  }

  async login(username: string, password: string) {
    const response = await this.http.post<LoginResponse>('/auth/token', {
      username,
      password
    }).toPromise();

    this.authDataService.permissions = response.permissions;
    this.authDataService.user = response.user;
    this.authDataService.authToken = response.authToken;

    window.localStorage.setItem('authToken', JSON.stringify(response.authToken));
    window.localStorage.setItem('user', JSON.stringify(response.user));
    window.localStorage.setItem('permissions', JSON.stringify(response.permissions || ""));

    return response;
  }

  logout() {
    window.localStorage.removeItem('user');
    window.localStorage.removeItem('permissions');

    const token = JSON.parse(window.localStorage.getItem('authToken'));
    if (token && token.token) {
      let headers = new HttpHeaders();
      headers = headers.set('Authorization', `Bearer ${token.token}`);
      this.http.delete('/auth/token', {headers: headers}).subscribe(
        () => window.location.href = '/login',
        () => window.location.href = '/login'
      );
    } else {
      window.location.href = '/login';
    }
    window.localStorage.removeItem('authToken');

    this.authDataService.permissions = null;
    this.authDataService.user = null;
    this.authDataService.authToken = null;
  }

  async getStatus() {
    const response = await this.http.get<LoginResponse>('/auth/me').toPromise();

    this.authDataService.permissions = response.permissions;
    this.authDataService.user = response.user;
    this.authDataService.authToken = response.authToken;

    window.localStorage.setItem('authToken', JSON.stringify(response.authToken));
    window.localStorage.setItem('user', JSON.stringify(response.user));
    window.localStorage.setItem('permissions', JSON.stringify(response.permissions || ""));

    return response;
  }
}
