import { Injectable } from "@angular/core";
import { HttpClient } from "@angular/common/http";

import { Observable } from "rxjs";
import { EmailBatch } from "./email-batch.model";
import { Email } from "./email.model";
import { tap } from "rxjs/operators";

export interface EmailReply {
  emailId: string;
  recipientEmail: string;
  recipientName: string;
  subject: string;
  body: string;
}

@Injectable()
export class EmailsDataService {
  batches: EmailBatch[] = [];

  constructor(private http: HttpClient) {}

  getAllBatchesForOrganisation(
    organisationId: string,
    filter: string,
    filterId: string,
    page: number = 0
  ): Observable<EmailBatch[]> {
    const filterQueryString = filter === "organisation" ? "" : `&${filter}Id=${filterId}`;

    return this.http
      .get<EmailBatch[]>(`/organisations/${organisationId}/emails?page=${page}${filterQueryString}`)
      .pipe(tap(batches => (this.batches = batches)));
  }

  getBatchByIdForOrganisation(organisationId: string, batchId: string): Observable<EmailBatch> {
    return this.http.get<EmailBatch>(`/organisations/${organisationId}/emails/${batchId}`);
  }

  getThreadByIdForOrganisation(organisationId: string, batchId: string, threadId: string): Observable<Email[]> {
    return this.http.get<Email[]>(`/organisations/${organisationId}/emails/${batchId}/threads/${threadId}`);
  }

  markEmailAsRead(
    organisationId: string,
    batchId: string,
    threadId: string,
    emailId: string,
    read: boolean
  ): Observable<void> {
    return this.http.put<void>(`/organisations/${organisationId}/emails/${batchId}/threads/${threadId}`, {
      read,
      emailId
    });
  }

  replyToEmail(organisationId: string, batchId: string, threadId: string, data: EmailReply): Observable<void> {
    return this.http.post<void>(`/organisations/${organisationId}/emails/${batchId}/threads/${threadId}`, data);
  }

  sendEmail(organisationId: string, data: any): Observable<void> {
    return this.http.post<void>(`/organisations/${organisationId}/emails`, data);
  }
}
