import { Component, Input, OnInit } from "@angular/core";
import { NgbActiveModal } from "@ng-bootstrap/ng-bootstrap";
import { FormControl, FormGroup, ValidatorFn } from "@angular/forms";
import { markFormGroupTouched } from "@app/shared/form-group";
import { Observable, of } from "rxjs";
import { NotificationsService } from "angular2-notifications";
import * as moment from "moment";
import { format } from "date-fns";

export class FormModalInputOptions {
  constructor(
    public id: string,
    public label: string,
    public type: string = "text",
    public value: any = null,
    public validators: ValidatorFn[] = [],
    public extra: any = null
  ) {}
}

@Component({
  selector: "app-form-modal",
  templateUrl: "./form-modal.component.html",
  styleUrls: ["./form-modal.component.scss"],
})
export class FormModalComponent implements OnInit {
  @Input() title = "";
  @Input() description = "";
  @Input() confirmLabel = "Opslaan";
  @Input() fields: FormModalInputOptions[] = [];
  form: FormGroup = new FormGroup({});
  @Input() hasDeleteCallback = false;
  loading = false;
  @Input() onError: (value: any) => void;

  @Input() callback = (value: any): Observable<any> => of(true);
  @Input() deleteCallback = (value: any) => {};
  @Input() onSuccess = (value: any) => {};

  constructor(public activeModal: NgbActiveModal, private toast: NotificationsService) {}

  ngOnInit() {
    this.fields.forEach((field) => {
      const value =
        field.value === null
          ? ""
          : ["string", "boolean"].includes(typeof field.value)
          ? field.value
          : field.type === "datetime"
          ? ""
          : field.value.value;
      this.form.addControl(
        field.id,
        new FormControl(
          {
            value,
            disabled: field.extra && field.extra.disabled,
          },
          field.validators
        )
      );

      if (value && field.type === "datetime") {
        setTimeout(() => {
          this.form.get(field.id).setValue(format(new Date(value), "YYYY-MM-DDTHH:mm"));
        });
      }
    });
  }

  deletePressed() {
    this.activeModal.close();
    if (this.hasDeleteCallback) {
      this.deleteCallback(null);
    }
  }

  confirmPressed() {
    this.loading = true;
    markFormGroupTouched(this.form);
    if (this.form.valid) {
      const value = this.form.value;
      Object.keys(value).map((key, index) => {
        value[key] = typeof value[key] === "string" ? value[key].trim() : value[key];
      });
      this.callback(value).subscribe(
        (result) => {
          this.onSuccess(result);
          this.loading = false;
          this.activeModal.close(value);
        },
        (err) => {
          this.loading = false;
          if (this.onError) {
            this.onError(err);
          } else {
            this.toast.error("Er ging iets mis.", err.error.message);
          }
        }
      );
    }
  }
}
