import { Component, OnInit } from "@angular/core";
import { ActivatedRoute, Router } from "@angular/router";
import { NotificationsService } from "angular2-notifications";
import * as Color from "color";
import { FormBuilder, FormControl, FormGroup, Validators } from "@angular/forms";
import { PasswordValidators } from "@app/shared/password.validator";
import { HttpClient } from "@angular/common/http";
import { markFormGroupTouched } from "@app/shared/form-group";
import { OrganisationsDataService } from "@app/organisations/organisations-data.service";
import { SchoolsDataService } from "@app/schools/schools-data.service";
import { EventsDataService } from "@app/events/events-data.service";
import { Member } from "@app/shared/models/member.model";
import { filter, flatMap, take } from "rxjs/internal/operators";
import { forkJoin, Observable } from "rxjs";
import { Event } from "@app/shared/models/event.model";
import { DropdownOption } from "@app/shared/forms/form-input/form-input.component";
import { FormModalComponent, FormModalInputOptions } from "@app/shared/form-modal/form-modal.component";
import { NgbModal } from "@ng-bootstrap/ng-bootstrap";
import { CategoriesDataService } from "@app/categories/categories-data.service";
import { Category } from "@app/shared/models/category.model";
import { Address } from "@app/shared/models/address.model";
import { Group } from "@app/shared/models/group.model";
import { Person } from "@app/shared/models/person.model";
import { Organisation } from "@app/shared/models/organisation.model";
import { AuthDataService } from '../auth-data.service';

@Component({
  selector: "app-member-detail",
  templateUrl: "./member-detail.component.html",
  styleUrls: ["./member-detail.component.scss"]
})
export class MemberDetailComponent implements OnInit {
  organisation: Organisation = null;
  color: Color;
  currentUserId = "";
  member: Member;
  memberId = "";
  memberType = "";
  schoolId = "";

  canEdit = true;

  passwordForm: FormGroup = null;
  personForm: FormGroup = null;
  addressForm: FormGroup = null;

  events: Event[];
  eventsForm: FormGroup = null;

  categories: Category[];

  constructor(
    private router: Router,
    private route: ActivatedRoute,
    private formBuilder: FormBuilder,
    private toast: NotificationsService,
    private organisationsService: OrganisationsDataService,
    private schoolsService: SchoolsDataService,
    private categoriesService: CategoriesDataService,
    private modal: NgbModal,
    private eventsService: EventsDataService,
    private authDataService: AuthDataService,
    private http: HttpClient,
  ) {}

  ngOnInit() {
    this.organisation = this.organisationsService.currentOrganisation;
    this.color = this.organisation ? this.organisation.primaryColor : Color("#008cd6");

    this.currentUserId = this.authDataService.user.id;

    this.route.params.subscribe(params => {
      this.memberId = params.memberId;

      this.eventsService.getAllForOrganisation(this.organisation.id).subscribe(events => (this.events = events));
      this.categoriesService
        .getAllForOrganisation(this.organisation.id)
        .subscribe(categories => (this.categories = categories));

      this.route.data.subscribe(data => {
        this.memberType = data["type"];
        this.getMember();
      });
    });
  }

  getMember() {
    const urlComponents = this.router.url.split("/");

    switch (this.memberType) {
      case "organisationMember":
        this.organisation.id = urlComponents[2];
        this.organisationsService
          .getMembers(this.organisation.id)
          .pipe(
            flatMap(m => m),
            filter(m => m.id === this.memberId),
            take(1)
          )
          .subscribe(member => {
            this.member = member;
            this.initForms();
          });
        break;
      case "schoolMember":
        this.schoolId = urlComponents[4];
        this.schoolsService.getMember(this.organisation.id, this.schoolId, this.memberId).subscribe(member => {
          this.member = member;
          this.initForms();
        });
        break;
    }
  }

  initForms() {
    this.passwordForm = new FormGroup(
      {
        password: new FormControl("", Validators.required),
        passwordRepeat: new FormControl("", Validators.required)
      },
      [PasswordValidators.matchPassword]
    );

    const person = this.member.person || new Person();

    this.personForm = new FormGroup({
      firstName: new FormControl(person.firstName),
      insertion: new FormControl(person.insertion),
      lastName: new FormControl(person.lastName),
      phoneNumber: new FormControl(person.phoneNumber)
    });

    const address = person.address || new Address();

    this.addressForm = new FormGroup({
      postcode: new FormControl(address.postcode, [Validators.pattern(/^( )*[0-9]{4}[a-zA-Z]{2}( )*$/)]),
      number: new FormControl(address.number, [Validators.pattern(/^[0-9]+$/)]),
      numberAddition: new FormControl(address.numberAddition),
      street: new FormControl({
        value: address.street,
        disabled: true
      }),
      city: new FormControl({
        value: address.city,
        disabled: true
      })
    });

    if (this.member.accessLevel !== "schoolStudent" && this.member.lastActiveAt) {
      this.canEdit = false;
      this.personForm.disable();
      this.addressForm.disable();
    } else {
      this.canEdit = true;
    }

    if (["organisationGuide", "schoolGuide", "schoolStudent"].includes(this.member.accessLevel)) {
      this.eventsForm = this.formBuilder.group({
        events: this.buildEventForm(this.events)
      });
    }
  }

  changePassword() {
    markFormGroupTouched(this.passwordForm);

    if (this.passwordForm.valid) {
      this.http.put(`/users/${this.memberId}`, this.passwordForm.value).subscribe(
        () => this.toast.success("Nieuw wachtwoord succesvol ingesteld"),
        error => this.toast.error("Er ging iets mis.", error.error.message ? "Huidige wachtwoord incorrect." : "")
      );
    }
  }

  _cleanObject(object: any) {
    for (const propName in object) {
      if (object[propName] === null || object[propName] === undefined) {
        delete object[propName];
      }
    }

    return object;
  }

  updatePerson() {
    if (this.personForm.dirty || this.addressForm.dirty) {
      markFormGroupTouched(this.personForm);
      markFormGroupTouched(this.addressForm);

      const person = this._cleanObject({ ...this.personForm.value });
      const address = this._cleanObject({ ...this.addressForm.value });

      if (this.personForm.valid && this.addressForm.valid) {
        let updateCall: Observable<Member> = null;
        switch (this.memberType) {
          case "organisationMember":
            updateCall = this.organisationsService.updateMember(this.organisation.id, {
              id: this.member.id,
              person: { ...person, address: address === {} ? undefined : address }
            });
            break;
          case "schoolMember":
            updateCall = this.schoolsService.updateMember(this.organisation.id, this.schoolId, {
              id: this.member.id,
              person: { ...person, address: address === {} ? undefined : address }
            });
            break;
        }
        if (updateCall) {
          updateCall.subscribe(
            member => {
              this.member.person = member.person;
              this.toast.success("Aanpassing opgeslagen");
            },
            err => this.toast.error(err.error.message)
          );
        }
      }
    }
  }

  updatePassword() {
    if (this.passwordForm.dirty) {
      markFormGroupTouched(this.passwordForm);

      if (this.passwordForm.valid) {
        this.schoolsService
          .updateMember(this.organisation.id, this.schoolId, {
            id: this.member.id,
            password: this.passwordForm.value.password
          })
          .subscribe(
            member => {
              this.toast.success("Wachtwoord aangepast");
            },
            err => this.toast.error(err.error.message)
          );
      }
    }
  }

  buildEventForm(events: Event[]) {
    return this.formBuilder.array(
      this.events.map(e => {
        return this.formBuilder.control(this.member.events.findIndex(me => me.id === e.id) !== -1);
      })
    );
  }

  onSubmitEvents() {
    markFormGroupTouched(this.eventsForm);

    const values = this.eventsForm.value.events;
    const remove = this.events
      .filter((e, i) => this.member.events.findIndex(me => me.id === e.id) !== -1 && !values[i])
      .map(e => e.id);

    const add = this.events
      .filter((e, i) => this.member.events.findIndex(me => me.id === e.id) === -1 && values[i])
      .map(e => e.id);

    const parallel = [];
    parallel.push(
      remove.map(eventId =>
        this.eventsService.deleteMember(
          this.organisation.id,
          eventId,
          this.memberId,
          this.memberType === "schoolMember" ? this.schoolId : ""
        )
      )
    );

    parallel.push(
      add.map(eventId =>
        this.eventsService.addMember(
          this.organisation.id,
          eventId,
          this.member,
          this.memberType === "schoolMember" ? this.schoolId : ""
        )
      )
    );

    const calls = parallel.length > 1 ? [].concat(...parallel) : parallel;

    if (calls.length > 0) {
      console.log(parallel, calls);
      forkJoin(calls).subscribe(
        () => {},
        () => this.toast.error("Er ging iets mis."),
        () => {
          this.toast.success("Deelname opgeslagen.");
          this.getMember();
        }
      );
    }
  }

  findIndexForEvent(event: Event) {
    return this.events.findIndex(e => e.id === event.id);
  }

  editCategoriesPressed() {
    const modalRef = this.modal.open(FormModalComponent);
    const component = modalRef.componentInstance as FormModalComponent;
    component.title = "Categorieën aanpassen";
    component.confirmLabel = "Aanpassen";
    component.fields = [
      new FormModalInputOptions(
        "categories",
        "Categorieën",
        "multiselect",
        {
          value: this.member.categories.map(cat => cat.id),
          options: this.categories.map(category => new DropdownOption(category.id, category.name))
        },
        [Validators.required],
        {
          size: this.categories.length
        }
      )
    ];

    component.callback = (result: any) =>
      this.schoolsService.updateMember(this.organisation.id, this.schoolId, {
        id: this.member.id,
        categories: result.categories
      });
    component.onSuccess = () => {
      this.toast.success("Categorieén aangepast");

      this.getMember();
    };
  }

  getEventNameFromId(eventId: string) {
    return this.events.find(e => e.id === eventId).name;
  }

  filterByUpcomingEvent(groups: Group[]) {
    return groups.filter(g => this.events.findIndex(e => e.id === g.eventId) !== -1);
  }

  groupClicked(group: Group) {
    this.router.navigate(["/organisaties", this.organisation.id, "scholen", this.schoolId, "groepen", group.id]);
  }
}
