import { Component, Input, OnInit } from '@angular/core';
import { FormModalComponent } from '@app/shared/form-modal/form-modal.component';
import { Address } from '@app/shared/models/address.model';
import { Location } from '@app/shared/models/location.model';
import { Company } from '@app/shared/models/company.model';
import { School } from '@app/shared/models/school.model';
import { NgbActiveModal } from '@ng-bootstrap/ng-bootstrap';
import { DropdownOptionGroup } from '@app/shared/forms/form-input/form-input.component';
import { Slot } from '@app/shared/models/slot.model';
import { Group } from '../models/group.model';

@Component({
  selector: 'app-route-modal',
  templateUrl: './route-modal.component.html',
  styleUrls: ['./route-modal.component.scss'],
})
export class RouteModalComponent extends FormModalComponent implements OnInit {
  @Input() destinations: (School | Company | Location)[] = [];
  @Input() slots: Slot[] = [];
  @Input() groupUrl: string = '';

  degreesToRadians(degrees: number) {
    return (degrees * Math.PI) / 180;
  }

  distanceInKmBetweenAddresses(addressA: Address, addressB: Address) {
    let lat1 = addressA.position.coordinates[1];
    let lat2 = addressB.position.coordinates[1];
    const lon1 = addressA.position.coordinates[0];
    const lon2 = addressB.position.coordinates[0];
    const earthRadiusKm = 6371;
    const accuracyMultiplier = 1.2;

    const dLat = this.degreesToRadians(lat2 - lat1);
    const dLon = this.degreesToRadians(lon2 - lon1);

    lat1 = this.degreesToRadians(lat1);
    lat2 = this.degreesToRadians(lat2);

    const a =
      Math.sin(dLat / 2) * Math.sin(dLat / 2) +
      Math.sin(dLon / 2) * Math.sin(dLon / 2) * Math.cos(lat1) * Math.cos(lat2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    return Math.round(earthRadiusKm * c * accuracyMultiplier * 10) / 10;
  }

  getDistanceForIndex(index: number) {
    const fields = Object.values(this.form.value);
    const locationA = this.destinations.find((l: School | Company | Location) => l.id === fields[index]);
    const locationB = this.destinations.find((l: School | Company | Location) => l.id === fields[index - 1]);

    if (!locationA || !locationB) {
      return '';
    }
    const addressA = locationA.address;
    const addressB = locationB.address;
    if (!addressA && !addressB) {
      return '<small>Adres ontbreekt voor beide bestemmingen</small>';
    }
    if (!addressA) {
      return `<small>Adres ontbreekt voor ${locationA.name}</small>`;
    }
    if (!addressB) {
      return `<small>Adres ontbreekt voor ${locationB.name}</small>`;
    }

    return `<small class='distance-label'>Geschatte afstand: ${this.distanceInKmBetweenAddresses(
      addressA,
      addressB
    )}km</small>`;
  }

  hasLogo(id: string) {
    return (this.destinations.find((d) => d.id === id) as Company).logo;
  }

  getLogoForId(id: string) {
    const destination = this.destinations.find((d) => d.id === id) as Company;
    if (destination.logo) {
      return destination.logo;
    }
    return '';
  }

  getTotalDistance() {
    const values = Object.values(this.form.value).filter((v: string) => v.length > 0);
    const addresses = values.map((v) => this.destinations.find((d) => d.id === v).address).filter((a) => a);
    let distance = 0;
    for (let i = 1; i < addresses.length; ++i) {
      distance += this.distanceInKmBetweenAddresses(addresses[i], addresses[i - 1]);
    }

    const url = `https://www.google.com/maps/dir/${addresses
      .map((a) => [...a.position.coordinates].reverse().join(','))
      .join('/')}`;
    return `<a target="_blank" href="${url}" class='distance-label'>Geschatte totale afstand: ${distance}km</a>`;
  }
}
