/*
 ******************************************************************************
 *  Copyright 1996 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink time conversion utilities.
 *
 * File: dl_time.c
 *
 * This file contains utility routines for time conversion from FactoryLink format
 * to 'C' format. The FactoryLink time starts at 1/1/1980 whereas the 'C' time starts 
 * at 1/1/1970.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <flib.h>
#include <fl_utils.h>             /* FactoryLink definitions */


static u32 _is_leap_year( u16);


/*-----------------------------------------------------------------------------
 *
 * function:                 dl_time2kdt
 *   
 *
 * arguments:
 *   u32      ltime          SECTIME as of 1/1/1980 (USDATA style)
 *   KDT      *kdtp          Pointer to a KDT structure 
 *
 * purpose:
 *   kdttime transforms the ltime into a kdt structure for use in the fmt_time()
 *   function. It is used instead of the gmtime() function which is not supported
 *   on VMS and VMA.
 *
 * returns:
 *   the individual values for a time.
 *
 *-----------------------------------------------------------------------------
 */

void dl_time2kdt( KDT *kdtp, u32 ltime)
{

  /* NUL,JAN,FEB,MAR,APR,MAY,JUN,JUL,AUG,SEP,OCT,NOV,DEC */
  int	days_of_leap[] = {   0,  0, 31, 60, 91,121,152,182,213,244,274,305,335 };
  int	days_of_year[] = {   0,  0, 31, 59, 90,120,151,181,212,243,273,304,334 };
  int	days_per_month[] = { 0, 31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30,31 };
  u32	leap_yr;
  int	doy;


	kdtp->year = 1980;
	kdtp->weekday = (u16)(ltime / SECONDS_PER_DAY + 2) % 7;

	leap_yr = _is_leap_year( kdtp->year );

	while ( ltime >= (SECONDS_PER_YEAR + SECONDS_PER_DAY * leap_yr) )
	{
		ltime -= SECONDS_PER_YEAR;

		if ( leap_yr )
			ltime -= SECONDS_PER_DAY;
	
		kdtp->year ++;
		leap_yr = _is_leap_year( kdtp->year);
	}

	doy    = (int)(ltime / SECONDS_PER_DAY);

	ltime -= doy * SECONDS_PER_DAY;

	kdtp->month = 12;
  
	if ( leap_yr )
	{
		days_per_month[2] = 29;

		while ( days_of_leap[kdtp->month] > doy)
			kdtp->month--;
	}
	else
		while ( days_of_year[kdtp->month] > doy)
			kdtp->month--;

	if ( leap_yr )
		kdtp->day = 1 + doy - days_of_leap[kdtp->month];
	else
		kdtp->day = 1 + doy - days_of_year[kdtp->month];

	kdtp->hours = (u16)(ltime / 3600L);
	ltime -= kdtp->hours * 3600L;

	kdtp->minutes = (u16)(ltime / 60L);
	ltime -= kdtp->minutes * 60L;

	kdtp->seconds = (u16)ltime;
}

/*-----------------------------------------------------------------------------
 *
 * function:                 dl_kdt2time
 *   
 *
 * arguments:
 *   KDT      *kdtp          Pointer to a KDT structure 
 *   u32      *sectime       SECTIME as of 1/1/1980 (USDATA style)
 *
 * purpose:
 *   kdt_mktime transforms the kdt structure into sectime.
 *   It is used instead of the mttime() function which is not equal on
 *   all platforms.
 *
 * returns:
 *   the sectime.
 *
 *-----------------------------------------------------------------------------
 */

u32 dl_kdt2time( KDT *kdtp)
{


  /* NUL,JAN,FEB,MAR,APR,MAY,JUN,JUL,AUG,SEP,OCT,NOV,DEC */
  int	days_of_leap[] = {   0,  0, 31, 60, 91,121,152,182,213,244,274,305,335 };
  int	days_of_year[] = {   0,  0, 31, 59, 90,120,151,181,212,243,273,304,334 };
  u32	leap_yr;
  u32	doy;
  KDT	k;				 /* Local copy */
  u32	s = 0;		 /* Temp Sectime */
  int first = 0; /* need to mark the year 1980, get lost of jan the first */

	/*
	 * Convert the years into seconds
	 */
	k = *kdtp;
	s = 0;

	/*
	 * Remember if it is a Leap_year
	 */
	leap_yr = _is_leap_year( k.year );

	k.year -= 1980;

  /* remember if where in 1980 */
  if ( k.year )
    first = 1;

	while( k.year != 0)
	{
		if( _is_leap_year( k.year ))
			s += SECONDS_PER_LEAP;
		else
			s += SECONDS_PER_YEAR;

		k.year--;
	}

	/*
	 * Count howmany days have passed this year
	 */
	if( leap_yr)
		doy = days_of_leap[ k.month] + k.day;
	else
		doy = 1 + days_of_year[ k.month] + k.day;

  /* get lost of januari the first */
  if ((first) && (doy))
    doy--;

	s += doy * SECONDS_PER_DAY;

	/*
	 * Add the last part
	 */
	s += k.hours * 3600L;
	s += k.minutes  * 60L;
	s += k.seconds;

	return s;
}

/*-----------------------------------------------------------------------------
 *
 * function:                 _is_leap_year
 *   
 *
 * arguments:
 *   u16      year           year to be checked
 *
 * purpose:
 *   is_leap_year returns if the year is a leap year or not
 *
 * returns:
 *   1 if the year is a leap year, 0 if not
 *
 *-----------------------------------------------------------------------------
 */
char *dl_ctime( KDT *kdt) {

	static char date[ 120],
							day[7][4] = {	"Sun",
														"Mon",
														"Tue",
														"Wed",														
														"Thu",
														"Fri",
														"Sat"},
							month[12][4] = {	"Jan",
																"Feb",
																"Mar",
																"Apr",
																"May",
																"Jun",
																"Jul",
																"Aug",
																"Sep",
																"Oct",
																"Nov",
																"Dec"};

	sprintf( date, "%s %s %02d %02d:%02d:%02d %d",
								day[ kdt->weekday],
								month[ kdt->month -1],
								kdt->day,
								kdt->hours,
								kdt->minutes,
								kdt->seconds,
								kdt->year);

	return date;
}

/*-----------------------------------------------------------------------------
 *
 * function:                 _is_leap_year
 *   
 *
 * arguments:
 *   u16      year           year to be checked
 *
 * purpose:
 *   is_leap_year returns if the year is a leap year or not
 *
 * returns:
 *   1 if the year is a leap year, 0 if not
 *
 *-----------------------------------------------------------------------------
 */
static u32 _is_leap_year( u16 year)
{

	if (!(year % 4) && (year % 100 || !(year % 400)))
		return 1L;
	else
		return 0L;
}
