/*
 ******************************************************************************
 *  Copyright 1996 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink synchronization utilities.
 *
 * File: fl_sync.c
 *
 * This file contains utility routines for the synchronization of the Real-Time 
 * Database of FactoryLink in case of multi-threaded processes. These routines
 * are platform specific and in this case for Windows-NT.
 *
 */
#ifdef UNIX
#include  <pthread.h>
#endif

#ifdef WIN32

#define NOMSG             // typedef MSG and associated routines
#define _OLE2_H_          // no ole

#include  <windows.h>
#include  <stdio.h>
#include  <stdlib.h>

#undef ERROR

#endif

#include  <stdio.h>
#include  <stdlib.h>

#include  <flib.h>
#include  <fl_utils.h>                  /* FactoryLink definitions */
#include  <fl_sync.h>                   /* Synchronization definitions */

TASK_ID   *Main_task;
TASK_ID   Signal_task;

extern int        Critical_nested;

/*
 * Function     : fl_init_critical
 *
 * Description  : Initializes the critical section functionality.
 *
 */
int fl_init_critical( TASK_ID *main, TASK_ID **signal )
{
  /*
   * initialise the critical section object for use
   */
  if( InitializeCSection( &Critical_RTDB, 1) == -1)
    return ERROR;

#ifdef ECHANGING

  if( InitializeCSection( &Critical_ECHANGE, 1) == -1)
    return ERROR;

  if( InitializeCSection( &Critical_ESIGNAL, 1) == -1)
    return ERROR;

  /*
   * enable the RTDB access signal
   */
  if( fl_hold_sig( main->id, FLC_SIG_RTDB_ACCESS, 0) == ERROR)
    return ERROR;

#endif

#ifdef SIGNAL

  if( InitializeCSection( &Critical_SIGNAL, 1) == -1)
    return ERROR;

  /*
   * enable the RTDB access signal
   */
  if( fl_hold_sig( main->id, FLC_SIG_RTDB_ACCESS, 0) == ERROR)
    return ERROR;

#endif

  Main_task = main;

  /*
   * Fix for kernel problem: simultanious execution of fl_send_signal and fl_change_wait
   * is sometime a problem. Result is errors with 'FLSTATE' and sometimes a crash of
   * the entire FL system.
   * To solve this a signal to ourself is send with another task id! This will result
   * in a second antry in the process list !!!
   */

  memcpy( &Signal_task, Main_task, sizeof( TASK_ID ) );

#ifdef SIGNAL_TASK_ID

  /* create unique task name */

  sprintf( Signal_task.name, "SIGNAL%2d", main->dl_id);
  
  /* make the description */

  sprintf( Signal_task.desc, "Signal process for protocol driver '%s'", main->name );

  if (( Signal_task.id = fl_proc_init_app(  Signal_task.name,
                                            Signal_task.desc,
                                            Signal_task.fl_name,
                                            Signal_task.fl_domain,
                                            Signal_task.fl_user )) < 0 )
  {
    return ERROR;
  }

  *signal = &Signal_task;

#endif

  return GOOD;
}

/*
 * Function     : fl_exit_critical
 *
 * Description  : Release the ownership of the critical section. This
 *                function must be called when exiting the process.
 *
 */
int fl_exit_critical ( )
{
  /*
   * release the system resources allocated for this critical section.
   */
  if( DeleteCSection( &Critical_RTDB) == -1 )
    return ERROR;

#ifdef ECHANGING
  if( DeleteCSection( &Critical_ECHANGE ) == -1 )
    return ERROR;

  if( DeleteCSection( &Critical_ESIGNAL ) == -1 )
    return ERROR;
#endif

#ifdef SIGNAL
  if( DeleteCSection( &Critical_SIGNAL) == -1 )
    return ERROR;
#endif

  /*
   * Remove signal task from FL process list
   */
#ifdef SIGNAL_TASK_ID
  fl_proc_exit( Signal_task.id ); 
#endif

  return GOOD;
}

/*
 * Function     : fl_signal_critical
 *
 * Description  : This function sends a FactoryLink signal to this process
 *                to wake up a pending fl_wait function. This function must
 *                be called before trying to enter a critical section, thus
 *                before trying to access the RTDB in case the fl_wait 
 *                function has been used.
 */
int fl_signal_critical()
{
  /*
   * send the user defined signal to wake up the fl_wait function to our own
   * process
   */
  if( fl_send_sig( Signal_task.id, Main_task->name, FLC_SIG_RTDB_ACCESS ) != GOOD )
    return fl_error( Signal_task.id );

  return GOOD;
}

/*
 * Function     : fl_enter_critical
 *
 * Description  : This function must be called before accessing the FactoryLink
 *                real-time database. It tries to get hold of the RTDB and if this
 *                not succeeds directly the the thread will sleep until it gets
 *                the RTDB.
 */
int fl_enter_critical( int signal )
{
  /*
   * check if the criticalsection is already owned by this thread
   */
  if ( IsCSectionOwner() == GOOD)
    return GOOD;
  
  /*
   * try to enter the critical section. In case the critical section is owned
   * by another thread then wait till it is freed. First ceck if we are in event
   * driven mode. If this is the case then first signal the fl_wait to get out
   * of the critical section
   */
#ifdef SIGNAL
  if ( EnterCSection( &Critical_SIGNAL, S_INFINITE ) == -1 )
    return ERROR;

  if ( signal )
    fl_signal_critical();
#endif

  if ( EnterCSection( &Critical_RTDB, S_INFINITE ) == -1 )
    return ERROR;

  SetCSectionOwner();

#ifdef SIGNAL
  if ( !signal )
  {
    if ( LeaveCSection( &Critical_SIGNAL ) == -1 )
      return ERROR;
  }
  else
  {
    /*
     * in this case a sub thread has claimed the RTDB critical section. At this time
     * it is of no use to signal the main thread because the main thread has already
     * left its fl_change_wait. However the signal must be blocked so that wait functions
     * in the sub thread will not be affected.
     */
    fl_hold_sig( Main_task->id, FLC_SIG_RTDB_ACCESS, ON);
  }
#endif

  return GOOD;
}

/*
 * Function     : fl_leave_critical
 *
 * Description  : Exits the critical section and gives control to another thread
 *                to enter the critical section.
 *
 */
int fl_leave_critical( int signal )
{
  /*
   * try to reset the owner. In case an error occurs then the section has been 
   * nested so we can leave immediately.
   */
  if( ResetCSectionOwner() == ERROR) 
    return ERROR;

#ifdef SIGNAL
  if ( signal )
  {
    /*
     * The sub thread leaves the critical section so enable the signal again
     */
    fl_hold_sig( Main_task->id, FLC_SIG_RTDB_ACCESS, OFF);
  }
#endif

  /*
   * release the ownership of the critical section.
   */

  if( LeaveCSection( &Critical_RTDB) == -1 )
    return ERROR;

#ifdef SIGNAL
  if ( signal )
  {
    if( LeaveCSection( &Critical_SIGNAL) == -1 )
      return ERROR;
  }
#endif

  return GOOD;
}
