/*
 ******************************************************************************
 *  Copyright 1994 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink general utilities.
 *
 * File: fl_utils.c
 *
 * This file contains utility routines for initialisation and protection
 * of DeltaLink FactoryLink tasks  
 *
 */

#ifdef UNIX
#include <sys/types.h>
#include <signal.h>
#include <sys/time.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <flib.h>
#include <fl_utils.h>

static	int	a_flag;

static	void	sig_alarm( int);

/*
 *-----------------------------------------------------------------------------
 * Function/Type: dl_sleep
 * Description  : suspends a task from executing for a period of time to give
 *								other tasks a chance of running. 
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : mode of the running task
 *
 *-----------------------------------------------------------------------------
 */

void fl_sleep( ulong msecs) {

#ifdef SCO
	if( msecs) nap( msecs);
#elif UNIX

	#ifndef	ITIMER_REAL
	if (msecs)
		sleep((int)(msecs+999)/1000);
	#else

	void	(*oldf)(int sig);
	struct	itimerval  newtime;
	struct	itimerval  oldtime;
	int	oldmask;

	if( msecs == 0)	return;

  /* block the receipt of alarm signals when setting out the timer */

	oldmask = sigblock( 1 << SIGALRM);

	/*
	 * Save the old signal handler for SIGALRM and set up a new one
	 * to sig_alarm().
	 */
	if( ( oldf = signal( SIGALRM, sig_alarm)) == SIG_ERR)
	  return;

	/*
	 * Set the time interval to 0 and the time value tv_sec and tv_usec
	 * fields appropriately.
	 */
	newtime.it_interval.tv_sec 	= 0;
	newtime.it_interval.tv_usec = 0;
	newtime.it_value.tv_sec 		= msecs / 1000;
	newtime.it_value.tv_usec 		= ( msecs % 1000) * 1000;

	timerclear( &oldtime.it_interval);
	timerclear( &oldtime.it_value);

	a_flag = 0;

	/* Now sleep until SIGALRM is delivered to sig_alarm(). */

	if( setitimer( ITIMER_REAL, &newtime, &oldtime) != -1 )	{

		while( !a_flag)	sigpause( 0);
	}

	/* Restore the old signal handler for SIGALRM. */

	signal(SIGALRM, oldf);

	sigblock( oldmask);
	#endif	/* ITIMER_REAL */

#else
	fl_sleep( msecs);
#endif	
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: sig_alarm
 * Description  : In case of unix the dl_sleep makes of of a alarm timer. This
 *								function will be called when the timer expires.
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : mode of the running task
 *
 *-----------------------------------------------------------------------------
 */

static	void sig_alarm( int sig) {

	a_flag = 1;
}
