/*
 ******************************************************************************
 *  Copyright 1994 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink general utilities.
 *
 * File: dl_utils.c
 *
 * This file contains utility routines for initialisation and protection
 * of DeltaLink FactoryLink tasks  
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <flib.h>
#include <flker.h>
#include <fl_utils.h>             /* FactoryLink definitions */

#ifdef BUTTON
#include <button.h>     /* Protocol Driver protection */
#endif

#include <encrypt.h>

#include <modules.h>    /* module and protection definition */

#define BUTTON_CHK_INTERVAL  15 * 60
#define BUTTON_MAX_MISS      10
time_t   button_chk_time;
unsigned button_miss = BUTTON_MAX_MISS;

#define TICKPERSEC      18L

#ifdef UNIX
#define DEMO_PERIOD 300L    /* demo period in minutes */

/*
 * In this case the clock function returns the number of microseconds
 * elapsed while this tak was running
 */
clock_t demo_time = DEMO_PERIOD * 60L * 1000000;

#else
#define DEMO_PERIOD 300L    /* demo period in minutes */

/*
 * In this case the clock function returns the number of ticks
 * elapsed while this tak was running
 */
clock_t demo_time = DEMO_PERIOD * ( 60L * CLOCKS_PER_SEC);
#endif


/*
 *-----------------------------------------------------------------------------
 * Function/Type: dl_init_task
 * Description  : initializes a FactoryLink task 
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : mode of the running task
 *
 *-----------------------------------------------------------------------------
 */

int dl_init_task( TASK_ID *task) {

  int          lite;
  OP_MODULE    module;                        /* module structure for option file */
  ANA          val;
  uint         index;
  short        bit = 0;


  /* set the cuurent timezone */

#ifndef UNIX
  if( !getenv( "TZ")) putenv("TZ=GMT0");
#endif

  tzset();

  /* initialize the different control tags of this task */
   
  task->k.e_ctrl.t_type = ( u16)FL_UNDEFINED;    /* control: start/stop (DIG or ANA) */
  task->k.e_stat.t_type = ( u16)FL_UNDEFINED;    /* self-reported status (ANA) */
  task->k.e_msg.t_type  = ( u16)FL_UNDEFINED;    /* self-reported message (MSG) */

  /* Acquire a task id */

  if( ( task->id = fl_proc_init_app( task->name,
                                     task->desc,
                                     fl_getvar( "FLNAME",   task->fl_name, MAX_PROC_NAME),
                                     fl_getvar( "FLDOMAIN", task->fl_domain, 8),
                                     fl_getvar( "FLUSER",   task->fl_user, MAX_USR_NAME))) < 0)
  {
    return DLE_FL_INIT;
  }

  /* set the task mode to polling */

  task->e_driven = 0;
  task->signal   = 0;

  /* get task environment */

  if( fl_get_env( task->id, &task->k) < 0)
  {
    return DLE_FL_ENV;
  }

	/*
	 * Retrieve the station ID number if specified on the command line
	 */
  if( strstr( task->k.e_cmd, "-LS") != NULL)
		task->station_id = atoi( strstr( task->k.e_cmd, "-LS") +3);

  /* initiliaze the debug and log levels */

  debug_log_init( task);

  /*
   * check if the task option is present in an option file or in a button
   */

  strcpy( module.name, task->name);
  module.id      = ( char)task->dl_id;
  module.version = ( char )( task->version >> 8);

  task->mode = 0;

	/* initialize the DeltaLink protection serial number to DEMO */

	strcpy( task->dl_serial, "DEMO_MODE");

#if defined( FLECS) 
  /* check for option bit */
  if (bit = dl_get_option_bit( task->dl_id))
  {

    /* check for the required option */
    if (!fl_check_bit( bit))
    {

      /* USDATA option present, normal running mode */

      if( !fl_check_bit( US_LITE_BIT))
        task->mode = LIC_LITE;
      else
        task->mode = LIC_FULL;

		  /* update the Task structure */

		  strncpy( task->dl_serial, fl_get_serial(), 8);
		  task->dl_serial[8] = '\0';
		  task->dl_version[ 0] = (char)(module.version && 0xff) + '0';
      task->dl_version[ 1] = '.';
		  task->dl_version[ 2] = (char)((module.version >> 8) && 0xff) + '0';
    }
    else
      bit = 0;
  }
#endif

  /* find deltalink protection if USDATA failed */
  if (!bit)
  {

#if !defined( FL413) 

	if( ( lite = pro_optfile( &module)) >= 0) {

		/* update the returned values on the Task structure */

		strncpy( task->dl_serial, module.dl_serial, 8);
		task->dl_serial[8] = '\0';
		task->dl_version[ 2] = module.dl_version[ 0];
    task->dl_version[ 1] = '.';
		task->dl_version[ 0] = module.dl_version[ 1];

    /* check if the task is limited to lite */

    if( lite)
      task->mode = PRT_OPTFILE | LIC_LITE;
    else
      task->mode = PRT_OPTFILE | LIC_FULL;
  }
  else
#endif

#ifdef BUTTON

	if( ( lite = button_init( task->dl_id, task->version)) >= 0) {

		/* update the returned values on the Task structure */

		strncpy( task->dl_serial, "BUTTON", 8);

		/* check if the task is limited to lite */

		if( lite)
			task->mode = PRT_BUTTON | LIC_LITE;
		else
			task->mode = PRT_BUTTON | LIC_FULL;
	}
	else {
#else
	{
#endif

		/* update the returned values on the Task structure */

		strncpy( task->dl_serial, module.dl_serial, 8);
		task->dl_serial[8] = '\0';
		task->dl_version[ 0] = module.dl_version[ 0];
    task->dl_version[ 1] = '.';
		task->dl_version[ 2] = module.dl_version[ 1];

		/* run in demo mode */

		task->mode = LIC_DEMO;

		demo_time += clock();

		/*
		 * check if the task has been started previously
		 */

		index = 0;
		if( fl_chread( task, &task->k.e_stat, 1, &index, &val) == CHANGED) {

			/* restore the change flag */

			fl_set_chng( task->id, &task->k.e_stat, 1);

			dl_task_banner( task);

			return DLE_DEMO_INIT;
		}
	}
}
	dl_task_banner( task);

	return GOOD;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: dl_task_banner
 * Description  : Print a task banner with start-up information
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : mode of the running task
 *
 *-----------------------------------------------------------------------------
 */

int dl_task_banner( TASK_ID *task) {

	printf("\n");

	debug_log( task, "START of %s: %s Version %d.%d\n\n", 1,
         task->name,
         task->desc,
				 task->version >> 8,
				 task->version & 0x00FF);

	debug_log( task, "\r FLDOMAIN\t<%s>\n FLNAME\t\t<%s>\n FLUSER\t\t<%s>\n FLAPP\t\t<%s>\n FLBIN\t\t<%s>\n", 1,
         task->fl_domain,
         task->fl_name,
         task->fl_user,
         task->k.e_adir,
         task->k.e_pdir);

	debug_log( task, "\r FL serial# <%s>, Version %d.%d\n DL serial# <%s>, Version %s\n", 1,
         fl_get_serial(),
         fl_get_version() >> 8,
         fl_get_version() & 0x00FF,
         task->dl_serial,
         task->dl_version);

	return GOOD;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: dl_test_term_flag
 * Description  : checks the running mode of the task and checks if the task may
 *                remain active
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : task ON or task OFF
 *
 *-----------------------------------------------------------------------------
 */

int dl_test_term_flag( TASK_ID *task) {


  /*
   * check the FactoryLink termination flag
   */
  if( task->e_driven) {

    if( task->signal == FLC_SIG_TERM_FLAG_SET ||
        fl_test_term_flag( task->id) == ON
      )
      return ON;

		/*
		 * reset the the mode to polling, fl_wait_event turns automatically
		 * the event mode on
		 */
		task->e_driven = 0;
		task->signal   = 0;
	}
	else {

		if( fl_test_term_flag( task->id) == ON) return ON;

		/* go into a sleep of 1 millisecond */

#ifdef UNIX
		dl_sleep( 50L );
#else
		dl_sleep( 1L );
#endif
	}


	/*
   * check in which mode this task runs
   */
  if( task->mode & LIC_DEMO) {

    /*
     * check the time the task is running
     */
    if( clock() > demo_time) return ON;

  }
#ifdef BUTTON
	else if( task->mode & PRT_BUTTON) {

		/*
		 * check the presence of the button after a period of time
		 */
		if( time( NULL) > button_chk_time) {

      if( button_check() != TRUE) {

        if( !button_miss) return ON;

        /*
   * notify the user that the DeltaLink key is missing
   */

  fl_status( task,
     "NO_KEY",
     FLS_ERROR,
     button_miss * (BUTTON_CHK_INTERVAL/60));

  button_miss--;
      }
      else
        button_miss = BUTTON_MAX_MISS;
  
      button_chk_time = time( NULL) + BUTTON_CHK_INTERVAL;
    }
  }
#endif

  return OFF;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: dl_exit_task
 * Description  : exits a FactoryLink task 
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : Pointer to task structure
 *
 * Exits with   : exit OK or not
 *
 *-----------------------------------------------------------------------------
 */
int dl_exit_task( TASK_ID *task, int error) {


	/*
	 * check in which mode this task runs
	 */
#ifdef BUTTON
	if( task->mode & PRT_BUTTON) {

		/*
		 * leave the button protection mechanism
		 */
		button_exit();
	}
#endif

	/*
	 * normal shutdown
	 */
	fl_shutdown( task, error);

	/*
	 * initialize the debug log functionality
	 */
	debug_log_exit( task);

	return GOOD;
}


/*
 *-----------------------------------------------------------------------------
 * Function/Type: short dl_get_option_bit( int dl_id)
 * Description  : Find USDATA protection bit number, with DeltaLink task id.
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 * Enter with  : DeltaLink task id
 *
 * Exits with   : USDATA protection bit, zero for not found.
 *
 *-----------------------------------------------------------------------------
 */
short dl_get_option_bit( int dl_id)
{


  switch (dl_id)
  {

    case IOXLATOR:
      return 64;

    case S_3964R:
      return 103;

    case SAPI_S7:
      return 106;

    case SINEC_H1:
      return 104;

    case PROFIBUS:
      return 105;

    default:
      return 0;
  }

  return 0;
}