/*
 ******************************************************************************
 *  Copyright 1996 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink synchronization utilities.
 *
 * File: sco_sync.c
 *
 */

#include <pthread.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#include <errno.h>

#include <flib.h>                     
#include <fl_sync.h>                     /* Synchronization definitions */
#include <fl_utils.h>                

/*
 * Define the critical section objects
 */
CRITICAL_SECTION  Critical_RTDB;
CRITICAL_SECTION  Critical_SIGNAL;

pthread_t Critical_owner;
int       Critical_nested;


/*
 * Function     : InitializeCSection
 *
 * Description  : Initializes the critical section functionality.
 *
 */
int InitializeCSection( CRITICAL_SECTION *CriticalSection, int n) {

  /* 
   * create the semaphore for use with this critical section
   */
  if( ( CriticalSection->semid = semget( IPC_PRIVATE, 1, 0x01ff | IPC_CREAT )) == -1)
    return -1;

  /*
   * initialize the semaphore to value 0
   */
  if( semctl( CriticalSection->semid, 0, SETVAL, n) == -1)
    return -1;

  return 0;
}

/*
 * Function     : DeleteCSection
 *
 * Description  : Deinitializes the critical section functionality.
 *
 */
int DeleteCSection( CRITICAL_SECTION *CriticalSection) {

  /*
   * remove the semaphore from the system
   */
  if( semctl( CriticalSection->semid, 1, IPC_RMID) == -1)
    return -1;

  CriticalSection->semid == -1;

  return 0;
}

/*
 * Function     : EnterCSection
 *
 * Description  : Deinitializes the critical section functionality.
 *
 */
int EnterCSection( CRITICAL_SECTION *CriticalSection, int mode )
{
  struct sembuf sops;

  sops.sem_num = 0;
  sops.sem_op  = -1;
  sops.sem_flg = 0;

  /*
   * try to enter the section by decrementing the semaphore with -1.
   */
  if( semop( CriticalSection->semid, &sops, 1 ) == -1)
    return -1;

  return 0;
}

/*
 * Function     : LeaveCSection
 *
 * Description  : Deinitializes the critical section functionality.
 *
 */
int LeaveCSection( CRITICAL_SECTION *CriticalSection) {

  struct sembuf sops;


  sops.sem_num = 0;
  sops.sem_op  = 1;
  sops.sem_flg = 0;

  /*
   * when leaving the critical section the value of the semaphore has
   * to be set to 1 again so other waiting entities fall through
   */
  if( semop( CriticalSection->semid, &sops, 1) == -1)
    return -1;

  return 0;
}

/*
 * Function     : ValidCSection
 *
 * Description  : Checks if the criticals section exists.
 *
 */
int ValidCSection( CRITICAL_SECTION *CriticalSection )
{
  /* 
   * create the semaphore for use with this critical section
   */
  if( CriticalSection->semid == -1 || CriticalSection->semid == 0 )
    return -1;

  return 0;
}

/*
 * Function     : SetCSectionOwner
 *
 * Description  : Check if this thread is the owner of the critical section
 *
 */
int SetCSectionOwner( ) 
{
  Critical_owner   = pthread_self();
  Critical_nested  = 0;

  return GOOD;
}

/*
 * Function     : ResetCSectionOwner
 *
 * Description  : Set the critical section state owned by nobody
 *
 */
int ResetCSectionOwner( )
{

  /* check first if this critical section has been nested */

  if( Critical_nested )
  {
    Critical_nested--;
    return ERROR;
  }
  else 
  {
    Critical_owner = 0;

    return GOOD;
  }
}

/*
 * Function     : IsCSectionOwner
 *
 * Description  : Check if this thread is the owner of the critical section
 *
 */
int IsCSectionOwner( )
{
  if ( pthread_self() == Critical_owner )
  {
    /* increment the nesting counter */

    Critical_nested++;

    return GOOD;
  }
  else
    return ERROR;
}
