/*
 ******************************************************************************
 *  Copyright 1995 DeltaLink bv. All Rights Reserved.
 ******************************************************************************
 *
 * DeltaLink FactoryLink functions for FactoryLink utilities.
 *
 * File: fl_func.c
 *
 */
#include  <stdio.h>
#include  <stdlib.h>
#include  <string.h>
#include  <stdarg.h>
#include  <ctype.h>
#include  <time.h>

#if defined( WIN)
#include  <dir.h>
#endif

#if defined( OS2) || defined( NT)
#include  <direct.h>
#endif

#include  <flib.h>
#include  <fl_utils.h>             /* FactoryLink definitions */

/*
 *-----------------------------------------------------------------------------
 * Function/Type: int find_event
 *
 * Description  : find an event in the event list. The right parameter returns
 *								the index in the event list where the event is placed or
 *								where it should be placed in case the event is not present in
 *								the list. The event list is sorted on the segment and offset
 *								of the tags.
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : event		tag which indicates the event
 *
 * Exits with : right		index into event list to event
 *							
 *							In case the event was present in the event list the function 
 *							returns zero otherwise not zero. If the event was not found
 *							the index indicates where the new event should be placed in the
 *						  event list
 *
 *-----------------------------------------------------------------------------
 */
int find_event( TAG *event, u16 *right) {

  int  equal;
	u16  left  = 0,
			 index;

	/* check first if there is a list of events */

	if( !_nr_events) {
	
	  *right = 0;
	  return -1;
	}

	*right = ( u16)( _nr_events -1);

	for(;;) {

	  /* calculate the index in the event array to check on */

		index = ( u16)( ( *right - left )/2 + left);
		equal = fl_cmptag( event, &e_taglist[ index]);

		/* check on which side we are from the index */

		if( equal > 0) 
			left = index;  					/* the left side to the new index */
		else if( equal < 0)
			*right = index; 				/* the right side to the new index */
		else
		  left = *right = index;  /* end criteria reached, tags are equal */

		/* 
		 * Check upon the end criterium. In case the end has been reached which means that
		 * two tags are left the check both tags.
		 */
		if( (*right - left) < 2) {

			/*
			 * check first the left side
			 */
			equal = fl_cmptag( event, &e_taglist[ left]);

			if( equal <= 0 ) {

				*right = left;
				break;
			}

			equal = fl_cmptag( event, &e_taglist[ *right]);

			if( equal > 0 )
				(*right)++;

	  	break;
		}
	}

	return equal;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_goto_begin
 *
 * Description  : go to the begin of a double linked list of type EVENT_FNC
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc			pointer to element to start with
 *
 * Exits with : pointer to the first element in the list
 *
 *-----------------------------------------------------------------------------
 */
EVENT_FNC *ll_goto_begin( EVENT_FNC *fnc) {

	EVENT_FNC *walk = fnc;

	if( walk)
    while( walk->prev != NULL) walk = fnc->prev;

	return walk;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_goto_end
 *
 * Description  : go to the end of a double linked list of type EVENT_FNC
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc			pointer to element to start with
 *
 * Exits with : pointer to the last element in the list
 *
 *-----------------------------------------------------------------------------
 */
EVENT_FNC *ll_goto_end( EVENT_FNC *fnc) {

	EVENT_FNC *walk = fnc;

	if( walk)
    while( walk->next != NULL) walk = fnc->next;

	return walk;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_count
 *
 * Description  : count the number of elements in the linked list
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc			pointer to an element in the linked list
 *
 * Exits with : number of elements element in the list
 *
 *-----------------------------------------------------------------------------
 */
 int ll_count( EVENT_FNC *fnc) {

  EVENT_FNC *f;
	int				count = 0;

	/* go first to the beginning of the linked list */

  if( ( f = ll_goto_begin( fnc)) == NULL)
	  return 0;

	/* walk the list to the end and count the elements */

	while( f != 0) {

		count++;
		f = f->next;
	}

  return count;
}


/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_insert
 *
 * Description  : insert an EVENT_FNC element in the linked list
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc			pointer to element where the ne element will be inserted
 *
 * Exits with : pointer to the new element
 *
 *-----------------------------------------------------------------------------
 */
EVENT_FNC *ll_insert( EVENT_FNC *old) {

	EVENT_FNC *new;

	/* allocate space for the new element */

	if( ( new = ( EVENT_FNC *)calloc( 1, sizeof( EVENT_FNC))) == NULL)
	  return NULL;

	if( old == NULL) return new;

	/* save the previous pointer of the old element */

	new->next = old;
	new->prev = old->prev;
	old->prev = new;

	if( new->prev) new->prev->next = new;

	return new;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_append
 *
 * Description  : append an EVENT_FNC element in the linked list
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc			pointer to element where the ne element will be appended
 *
 * Exits with : pointer to the new element
 *
 *-----------------------------------------------------------------------------
 */
EVENT_FNC *ll_append( EVENT_FNC *old) {

	EVENT_FNC *new;

	/* allocate space for the new element */

	if( ( new = ( EVENT_FNC *)calloc( 1, sizeof( EVENT_FNC))) == NULL)
	  return NULL;

	if( old == NULL) return new;

	/* save the previous pointer of the old element */

	new->next = old->next;
	new->prev = old->prev;
	old->prev = new;

	if( new->next) new->next->prev = new;

	return new;
}

/*
 *-----------------------------------------------------------------------------
 * Function/Type: ll_delete
 *
 * Description  : delete an EVENT_FNC element in the linked list
 *
 *------------------- Parameters, Variables, & Conditions ---------------------
 *
 * Enter with : fnc		pointer to element to delete from the list
 *
 * Exits with : pointer to the next element in the list
 *
 *-----------------------------------------------------------------------------
 */
EVENT_FNC *ll_delete( EVENT_FNC *fnc, EVENT_FNC **start) {

	EVENT_FNC *next = fnc->next;

  /* update pointer of the previous and the next element */

	if( fnc->prev) fnc->prev->next = fnc->next;
	if( fnc->next) fnc->next->prev = fnc->prev;

	/* set first the start to the linked list */

	if( ( *start = ll_goto_begin( fnc)) == NULL)
	  return NULL;

	/* check if the first element is to be deleted */

  if( *start == fnc) *start = next;

	/* free the memory location of the element */

	free( fnc);

	return next;
}
