/*
Module : ODBCWrappers.H
Purpose: Defines the interface for a set of C++ class which encapsulate ODBC 3.x. The classes are loosely 
         based on the design of the ATL OLEDB Consumer Template classes which are provided with 
         Microsoft Visual C++.
History: PJN / 24-09-2011 1. Initial creation
         PJN / 13-11-2011 1. Initial Public release.
         PJN / 16-11-2011 1. Updated code to use std::vector instead of std::auto_ptr for allocating SQLTCHAR
                          temp heap arrays. Thanks to Andrey Karpov for reporting this issue.
                          2. BrowseConnect method which returns a string& parameter now uses a TCHAR* parameter
                          instead of a SQLTCHAR* parameter for the first parameter
         PJN / 20-11-2011 1. Removed final occurrence of std::auto_ptr usage in CStatement::GetCursorName.
         PJN / 19-02-2013 1. Updated copyright details.
                          2. Updated code to provide default sensible values for the StrLen_or_IndPtr parameter
                          if it is not explicitly provided by client code.
                          3. Updated the sample app to shown examples of inserting into a table, iterating across
                          the rows in a table, updating a row in a table and deleting a row from a table.
                          4. Removed some accidental usage of MFC TRACE and ASSERT functions and replaced with 
                          ATL equivalents.
                          5. Pulling in MS SQL Server extensions header file sqlext.h and associated functionality
                          is now optional via a new CODBCWRAPPERS_MSSQL_EXTENSIONS preprocessor value
                          6. Sample app now does not link against MFC for demonstration purpose
         PJN / 21-02-2013 1. Addition of a new CODBCWRAPPERS_MFC_EXTENSIONS preprocessor value which changes the 
                          wrapper classes to more tightly integrate with MFC. Internally the classes will then use
                          the MFC collection classes and expose a CString interface 
                          2. Following a customer request the code should now be compilable in VC 6 for those 
                          diehards still stuck on this compiler.

Copyright (c) 2011 - 2013 by PJ Naughter (Web: www.naughter.com, Email: pjna@naughter.com)

All rights reserved.

Copyright / Usage Details:

You are allowed to include the source code in any product (commercial, shareware, freeware or otherwise) 
when your product is released in binary form. You are allowed to modify the source code in any way you want 
except you cannot modify the copyright details at the top of each module. If you want to distribute source 
code with your application, then you are only allowed to distribute versions released by the author. This is 
to maintain a single distribution point for the source code. 

*/


//////////////////////////// Macros / Includes ////////////////////////////////

#ifndef __ODBCWRAPPERS_H__
#define __ODBCWRAPPERS_H__

#ifndef CODBCWRAPPERS_EXT_CLASS
#define CODBCWRAPPERS_EXT_CLASS
#endif

#ifndef CODBCWRAPPERS_EXT_API
#define CODBCWRAPPERS_EXT_API
#endif

#pragma once

#ifndef __SQL
#pragma message("To avoid this message, please put sql.h in your pre compiled header (normally stdafx.h)")
#include <sql.h>
#endif

#ifdef CODBCWRAPPERS_MSSQL_EXTENSIONS
#ifndef __SQLEXT
#pragma message("To avoid this message, please put sqlext.h in your pre compiled header (normally stdafx.h)")
#include <sqlext.h>
#endif
#endif


//Handle old versions of the ODBC header files which do not define SQLLEN, SQLULEN & SQLSETPOSIROW
#ifndef SQLLEN
#ifdef _WIN64
typedef INT64 SQLLEN;
#else
#define SQLLEN SQLINTEGER
#endif
#endif

#ifdef _WIN64
typedef UINT64 SQLULEN;
#else
#define SQLULEN SQLUINTEGER
#endif

#ifdef _WIN64
typedef UINT64 SQLSETPOSIROW;
#else
#define SQLSETPOSIROW SQLUSMALLINT
#endif


//Handle old versions of VC which do not support ATLASSUME
#ifndef ATLASSUME
#define ATLASSUME ATLASSERT
#endif


#pragma comment(lib, "odbc32.lib")


#define DEFINE_ODBC_SQL_TYPE_FUNCTION(ctype, odbctype) \
	inline SQLSMALLINT _GetODBCSQLType(ctype&) throw ()\
	{ \
		return odbctype; \
	}
	
	inline SQLSMALLINT _GetODBCSQLType(BYTE[]) throw ()
	{
		return SQL_VARBINARY;
	}
	inline SQLSMALLINT _GetODBCSQLType(char[]) throw ()
	{
		return SQL_VARCHAR;
	}
	inline SQLSMALLINT _GetODBCSQLType(wchar_t[]) throw()
	{
		return SQL_WVARCHAR;
	}

	DEFINE_ODBC_SQL_TYPE_FUNCTION(char, SQL_CHAR)            
	DEFINE_ODBC_SQL_TYPE_FUNCTION(wchar_t, SQL_CHAR)            
	DEFINE_ODBC_SQL_TYPE_FUNCTION(bool, SQL_BIT)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(BYTE, SQL_TINYINT)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(short, SQL_SMALLINT)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(long, SQL_INTEGER)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(float, SQL_REAL)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(double, SQL_DOUBLE)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(__int64, SQL_BIGINT)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(TIMESTAMP_STRUCT, SQL_TYPE_TIMESTAMP)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(SQL_NUMERIC_STRUCT, SQL_NUMERIC)
	DEFINE_ODBC_SQL_TYPE_FUNCTION(GUID, SQL_GUID)

#define DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(ctype, defaultIndicatorValue) \
	inline void _GetODBCDefaultIndicatorValue(ctype&, SQLLEN* pDefaultIndicatorValue) throw ()\
	{ \
	  if (pDefaultIndicatorValue != NULL) \
		  *pDefaultIndicatorValue = defaultIndicatorValue; \
	}
	
	inline void _GetODBCDefaultIndicatorValue(BYTE[], SQLLEN* /*pDefaultIndicatorValue*/) throw ()
	{
	  ATLASSERT(FALSE); //If you pass a byte array, then you should explicitly specify the length yourself
	}
	inline void _GetODBCDefaultIndicatorValue(char[], SQLLEN* pDefaultIndicatorValue) throw ()
	{
		*pDefaultIndicatorValue = SQL_NTS; //Lets assume NULL terminated data for ASCII data
	}
	inline void _GetODBCDefaultIndicatorValue(wchar_t[], SQLLEN* pDefaultIndicatorValue) throw()
	{
		*pDefaultIndicatorValue = SQL_NTS; //Lets assume NULL terminated data for Unicode data
	}

	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(char, sizeof(char))            
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(wchar_t, sizeof(wchar_t))            
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(bool, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(BYTE, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(short, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(long, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(float, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(double, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(__int64, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(TIMESTAMP_STRUCT, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(SQL_NUMERIC_STRUCT, NULL)
	DEFINE_ODBC_DEFAULT_INDICATOR_VALUE_FUNCTION(GUID, NULL)
	
//The ODBC_CHECK_RETURN macro checks the return value from ODBC calls
#define ODBC_CHECK_RETURN(nRet, handle) \
  handle.ValidateReturnValue(nRet); \
  if ((nRet != SQL_SUCCESS) && (nRet != SQL_SUCCESS_WITH_INFO)) \
    return nRet;

//The ODBC_CHECK_RETURN_PTR macro checks the return value from ODBC calls
#define ODBC_CHECK_RETURN_PTR(nRet, handle) \
  handle->ValidateReturnValue(nRet); \
  if ((nRet != SQL_SUCCESS) && (nRet != SQL_SUCCESS_WITH_INFO)) \
    return nRet;

#define DEFINE_ODBC_C_TYPE_FUNCTION(ctype, odbctype) \
	inline SQLSMALLINT _GetODBCCType(ctype&) throw ()\
	{ \
		return odbctype; \
	}
	inline SQLSMALLINT _GetODBCCType(BYTE[]) throw ()
	{
		return SQL_C_BINARY;
	}
	inline SQLSMALLINT _GetODBCCType(char[]) throw ()
	{
		return SQL_C_CHAR;
	}
	inline SQLSMALLINT _GetODBCCType(wchar_t[]) throw()
	{
		return SQL_C_WCHAR;
	}

	DEFINE_ODBC_C_TYPE_FUNCTION(char, SQL_C_CHAR)            
	DEFINE_ODBC_C_TYPE_FUNCTION(wchar_t, SQL_C_WCHAR)
	DEFINE_ODBC_C_TYPE_FUNCTION(bool, SQL_C_BIT)
	DEFINE_ODBC_C_TYPE_FUNCTION(BYTE, SQL_C_TINYINT)
	DEFINE_ODBC_C_TYPE_FUNCTION(short, SQL_C_SSHORT)
	DEFINE_ODBC_C_TYPE_FUNCTION(long, SQL_C_SLONG)
	DEFINE_ODBC_C_TYPE_FUNCTION(float, SQL_C_FLOAT)
	DEFINE_ODBC_C_TYPE_FUNCTION(double, SQL_C_DOUBLE)
	DEFINE_ODBC_C_TYPE_FUNCTION(__int64, SQL_C_SBIGINT)
	DEFINE_ODBC_C_TYPE_FUNCTION(TIMESTAMP_STRUCT, SQL_C_TYPE_TIMESTAMP)
	DEFINE_ODBC_C_TYPE_FUNCTION(SQL_NUMERIC_STRUCT, SQL_C_NUMERIC)
	DEFINE_ODBC_C_TYPE_FUNCTION(GUID, SQL_C_GUID)

#define _ODBC_C_TYPE(data) _GetODBCCType((static_cast<_classtype*>(NULL))->data)
#define _ODBC_SQL_TYPE(data) _GetODBCSQLType((static_cast<_classtype*>(NULL))->data)
#define _ODBC_SIZE_TYPE(data) sizeof((static_cast<_classtype*>(NULL))->data)


#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
#define BEGIN_ODBC_PARAM_MAP(x) \
	public: \
	typedef x _classtype; \
	SQLRETURN _GetBindParametersEntries(CODBC::CStatement* pStmt, int* pnColumns, CArray<SQLLEN, SQLLEN>* pIndicators) \
	{ \
	  (pStmt); \
	  (pIndicators); \
		SQLSMALLINT ioType = SQL_PARAM_INPUT; \
    (ioType); \
		SQLRETURN nRet = SQL_SUCCESS; \
		if (pnColumns != NULL) \
		  *pnColumns = 0;
#else
#define BEGIN_ODBC_PARAM_MAP(x) \
	public: \
	typedef x _classtype; \
	SQLRETURN _GetBindParametersEntries(CODBC::CStatement* pStmt, int* pnColumns, std::vector<SQLLEN>* pIndicators) \
	{ \
	  (pStmt); \
	  (pIndicators); \
		SQLSMALLINT ioType = SQL_PARAM_INPUT; \
    (ioType); \
		SQLRETURN nRet = SQL_SUCCESS; \
		if (pnColumns != NULL) \
		  *pnColumns = 0;
#endif

#define END_ODBC_PARAM_MAP() \
		return nRet; \
	}

#define _ODBC_PARAM_ENTRY_CODE(ParameterNumber, ValueType, ParameterType, ColumnSize, DecimalDigits, ParameterValuePtr, BufferLength, StrLen_or_IndPtr, data) \
    if (pnColumns != NULL) \
      *pnColumns = ParameterNumber; \
    if (pStmt != NULL) \
    { \
      SQLLEN* pIndicator = StrLen_or_IndPtr; \
      if (pIndicator == NULL) \
      { \
        ATLASSUME(pIndicators != NULL); \
        SQLLEN& nParameterIndicator = (*pIndicators)[ParameterNumber - 1]; \
        _GetODBCDefaultIndicatorValue(data, &nParameterIndicator); \
        pIndicator = &nParameterIndicator; \
      } \
      nRet = pStmt->BindParameter(ParameterNumber, ioType, ValueType, ParameterType, ColumnSize, DecimalDigits, ParameterValuePtr, BufferLength, pIndicator); \
      ODBC_CHECK_RETURN_PTR(nRet, pStmt); \
    }

#define ODBC_PARAM_ENTRY(ParameterNumber, data) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), _ODBC_SIZE_TYPE(data), 0, &data, _ODBC_SIZE_TYPE(data), NULL, data);

#define ODBC_PARAM_LENGTH(ParameterNumber, data, length) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), length, 0, &data, length, NULL, data);

#define ODBC_PARAM_ENTRY_TYPE(ParameterNumber, data, cType, SQLType) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, cType, sqlType, _ODBC_SIZE_TYPE(data), 0, &data, _ODBC_SIZE_TYPE(data), NULL, data);

#define ODBC_PARAM_ENTRY_STATUS(ParameterNumber, data, status) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), _ODBC_SIZE_TYPE(data), 0, &data, _ODBC_SIZE_TYPE(data), &status, data);

#define ODBC_PARAM_LENGTH_STATUS(ParameterNumber, data, length, status) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), length, 0, &data, length, &status, data);

#define ODBC_PARAM_ENTRY_PS(ParameterNumber, nPrecision, data) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), _ODBC_SIZE_TYPE(data), nPrecision, &data, _ODBC_SIZE_TYPE(data), NULL, data);

#define ODBC_PARAM_PS_LENGTH(ParameterNumber, data, nPrecision, length) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), length, nPrecision, &data, length, NULL, data);

#define ODBC_PARAM_ENTRY_PS_TYPE(ParameterNumber, data, nPrecision, cType, SQLType) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, cType, sqlType, _ODBC_SIZE_TYPE(data), nPrecision, &data, _ODBC_SIZE_TYPE(data), NULL, data);

#define ODBC_PARAM_ENTRY_PS_STATUS(ParameterNumber, data, nPrecision, status) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), _ODBC_SIZE_TYPE(data), nPrecision, &data, _ODBC_SIZE_TYPE(data), &status, data);

#define ODBC_PARAM_PS_LENGTH_STATUS(ParameterNumber, data, nPrecision, length, status) \
  _ODBC_PARAM_ENTRY_CODE(ParameterNumber, _ODBC_C_TYPE(data), _ODBC_SQL_TYPE(data), length, nPrecision, &data, length, &status, data);

#define SET_ODBC_PARAM_TYPE(type) \
	ioType = type;

#define DEFINE_ODBC_COMMAND(x, szCommand) \
	static SQLTCHAR* GetDefaultCommand() throw () \
	{ \
		return reinterpret_cast<SQLTCHAR*>(szCommand); \
	}


#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  #define BEGIN_ODBC_COLUMN_MAP(x) \
	  public: \
	  typedef x _classtype; \
	  SQLRETURN _GetBindColumnEntries(CODBC::CStatement* pStmt, int* pnColumns, CArray<SQLLEN, SQLLEN>* pIndicators) \
	  { \
	    (pStmt); \
	    (pIndicators); \
		  SQLRETURN nRet = SQL_SUCCESS; \
		  if (pnColumns != NULL) \
		    *pnColumns = 0;
#else
  #define BEGIN_ODBC_COLUMN_MAP(x) \
	  public: \
	  typedef x _classtype; \
	  SQLRETURN _GetBindColumnEntries(CODBC::CStatement* pStmt, int* pnColumns, std::vector<SQLLEN>* pIndicators) \
	  { \
	    (pStmt); \
	    (pIndicators); \
		  SQLRETURN nRet = SQL_SUCCESS; \
		  if (pnColumns != NULL) \
		    *pnColumns = 0;
#endif

#define END_ODBC_COLUMN_MAP() \
		return nRet; \
	}

#define _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, TargetType, TargetValuePtr, BufferLength, StrLen_or_Ind) \
    if (pnColumns != NULL) \
      *pnColumns = ColumnNumber; \
    if (pStmt != NULL) \
    { \
      SQLLEN* pIndicator = StrLen_or_Ind; \
      if (pIndicator == NULL) \
      { \
        ATLASSUME(pIndicators != NULL); \
        SQLLEN& nParameterIndicator = (*pIndicators)[ColumnNumber - 1]; \
        pIndicator = &nParameterIndicator; \
      } \
      nRet = pStmt->BindCol(ColumnNumber, TargetType, TargetValuePtr, BufferLength, pIndicator); \
      ODBC_CHECK_RETURN_PTR(nRet, pStmt); \
    }

#define ODBC_COLUMN_ENTRY(ColumnNumber, data) \
  _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, _ODBC_C_TYPE(data), &data, _ODBC_SIZE_TYPE(data), NULL);

#define ODBC_COLUMN_ENTRY_LENGTH(ColumnNumber, data, length) \
  _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, _ODBC_C_TYPE(data), &data, length, NULL);

#define ODBC_COLUMN_ENTRY_TYPE(ColumnNumber, data, type) \
  _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, type, &data, _ODBC_SIZE_TYPE(data), NULL);

#define ODBC_COLUMN_ENTRY_STATUS(ColumnNumber, data, status) \
  _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, _ODBC_C_TYPE(data), &data, _ODBC_SIZE_TYPE(data), &status);

#define ODBC_COLUMN_ENTRY_LENGTH_STATUS(ColumnNumber, data, length, status) \
  _ODBC_COLUMN_ENTRY_CODE(ColumnNumber, _ODBC_C_TYPE(data), &data, length, &status);


/////////////////////////// Classes ///////////////////////////////////////////

namespace CODBC
{
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  typedef CString String;
#else
  #ifdef _UNICODE
    typedef std::wstring String; 
  #else
    typedef std::string String;
  #endif
#endif

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  void CODBCWRAPPERS_EXT_API MultiSZToStringVector(const SQLTCHAR* pszString, CArray<String, String&>& elements)
  {
    //Clear the vector
    elements.RemoveAll();
    
    //Add each element we find
    const TCHAR* pszValue = reinterpret_cast<const TCHAR*>(pszString);
    while (*pszValue != NULL)
    {
	  CString sValue(pszValue);		
      elements.Add(sValue);
      pszValue += (_tcslen(pszValue) + 1); 
    }
  }
#else
  void CODBCWRAPPERS_EXT_API MultiSZToStringVector(const SQLTCHAR* pszString, std::vector<String>& elements)
  {
    //Clear the vector
    elements.clear();
    
    //Add each element we find
    const TCHAR* pszValue = reinterpret_cast<const TCHAR*>(pszString);
    while (*pszValue != NULL)
    {
      elements.push_back(pszValue);
      pszValue += (_tcslen(pszValue) + 1); 
    }
  }
#endif


//Class which encapsulates a generic ODBC handle
class CODBCWRAPPERS_EXT_CLASS CHandle
{
public:
//Constructors / Destructors
  CHandle() : m_h(SQL_NULL_HANDLE),
              m_Type(0)
  {
  }

	CHandle(CHandle& h) : m_h(SQL_NULL_HANDLE),
	                      m_Type(0)
  {
    Attach(h.m_Type, h.Detach());
  }

	explicit CHandle(SQLSMALLINT Type, SQLHANDLE h) : m_h(h),
	                                                  m_Type(Type)
  {
  }

	~CHandle()
  {
    if (m_h != SQL_NULL_HANDLE)
      Close();
  }

	CHandle& operator=(CHandle& h)
  {
    if (this != &h)
    {
      if (m_h != SQL_NULL_HANDLE)
        Close();
      Attach(h.m_Type, h.Detach());
    }

    return *this;
  }

//Methods
  SQLRETURN Create(SQLSMALLINT Type, SQLHANDLE InputHandle)
  {
    //Validate our parameters
    ATLASSERT(m_h == SQL_NULL_HANDLE);
    
    m_Type = Type;
    return SQLAllocHandle(Type, InputHandle, &m_h);
  }

	void Close()
  {
    if (m_h != SQL_NULL_HANDLE)
    {
      SQLFreeHandle(m_Type, m_h);
      m_h = SQL_NULL_HANDLE;
      m_Type = 0;
    }
  }

	operator SQLHANDLE() const
  {
    return m_h;
  }

	void Attach(SQLSMALLINT Type, SQLHANDLE h)
  {
    //Validate our parameters
    ATLASSERT(m_h == SQL_NULL_HANDLE);
    
    m_Type = Type;
    m_h = h;
  }

	SQLHANDLE Detach()
  {
    SQLHANDLE h = m_h;
    m_h = SQL_NULL_HANDLE;
    m_Type = 0;
    return h;
  }
  
  SQLRETURN GetDiagField(SQLSMALLINT RecNumber, SQLSMALLINT DiagIdentifier, SQLPOINTER DiagInfoPtr, SQLSMALLINT BufferLength, SQLSMALLINT* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLGetDiagField(m_Type, m_h, RecNumber, DiagIdentifier, DiagInfoPtr, BufferLength, StringLengthPtr);
  }
  
  SQLRETURN GetDiagRec(SQLSMALLINT RecNumber, SQLTCHAR* SQLState, SQLINTEGER* NativeErrorPtr, SQLTCHAR* MessageText, SQLSMALLINT BufferLength, SQLSMALLINT* TextLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLGetDiagRec(m_Type, m_h, RecNumber, SQLState, NativeErrorPtr, MessageText, BufferLength, TextLengthPtr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  void GetDiagRecords(CArray<String, String&>& records)
  {
    //clear the vector
    records.RemoveAll();
    
    BOOL bContinue = TRUE;
    for (SQLSMALLINT i=1; bContinue; i++)
    {
      SQLTCHAR szState[6];
      szState[0] = _T('\0');
      SQLINTEGER nNativeError = 0;
      SQLTCHAR szMessageText[SQL_MAX_MESSAGE_LENGTH];
      szMessageText[0] = _T('\0'); 
      SQLSMALLINT nTextLength = 0;
      SQLRETURN nGetDiagRet = GetDiagRec(i, szState, &nNativeError, szMessageText, sizeof(szMessageText)/sizeof(SQLTCHAR), &nTextLength);
      if (nGetDiagRet == SQL_SUCCESS)
      {
        CString sLine;; 
        sLine.Format(_T("State:%s, Native Error Code:%d, Message Text:%s\n"), szState, nNativeError, szMessageText);
        records.Add(sLine);
      }
      else if (nGetDiagRet == SQL_NO_DATA)
        bContinue = FALSE;
    }
  }
#else
  void GetDiagRecords(std::vector<String>& records)
  {
    //clear the vector
    records.clear();
    
    BOOL bContinue = TRUE;
    for (SQLSMALLINT i=1; bContinue; i++)
    {
      SQLTCHAR szState[6];
      szState[0] = _T('\0');
      SQLINTEGER nNativeError = 0;
      SQLTCHAR szMessageText[SQL_MAX_MESSAGE_LENGTH];
      szMessageText[0] = _T('\0'); 
      SQLSMALLINT nTextLength = 0;
      SQLRETURN nGetDiagRet = GetDiagRec(i, szState, &nNativeError, szMessageText, sizeof(szMessageText)/sizeof(SQLTCHAR), &nTextLength);
      if (nGetDiagRet == SQL_SUCCESS)
      {
        TCHAR szLine[SQL_MAX_MESSAGE_LENGTH + 64]; 
        szLine[0] = _T('\0');
      #ifdef _stprintf_s
        _stprintf_s(szLine, sizeof(szLine)/sizeof(TCHAR), _T("State:%s, Native Error Code:%d, Message Text:%s\n"), szState, nNativeError, szMessageText);
      #else
        _stprintf(szLine, _T("State:%s, Native Error Code:%d, Message Text:%s\n"), szState, nNativeError, szMessageText);
      #endif
        records.push_back(szLine);
      }
      else if (nGetDiagRet == SQL_NO_DATA)
        bContinue = FALSE;
    }
  }
#endif

  void ValidateReturnValue(SQLRETURN nRet)
  {
  #ifdef _DEBUG
    if ((nRet != SQL_SUCCESS) && (nRet != SQL_SUCCESS_WITH_INFO))
    {
      //Report the bad return value
      ATLTRACE(_T("ODBC API Failure, Return Value:%d\n"), nRet);
    
      //Report all the diag records
      BOOL bContinue = TRUE;
      for (SQLSMALLINT i=1; bContinue; i++)
      {
        SQLTCHAR szState[6];
        szState[0] = _T('\0');
        SQLINTEGER nNativeError = 0;
        SQLTCHAR szMessageText[SQL_MAX_MESSAGE_LENGTH];
        szMessageText[0] = _T('\0'); 
        SQLSMALLINT nTextLength = 0;
        SQLRETURN nGetDiagRet = GetDiagRec(i, szState, &nNativeError, szMessageText, sizeof(szMessageText)/sizeof(SQLTCHAR), &nTextLength);
        if (nGetDiagRet == SQL_SUCCESS)
        {
        #ifdef CODBCWRAPPERS_MSSQL_EXTENSIONS  
        //Report on the MS SQL Extension values if necessary
          SQLLEN nRowNumber = 0;
          GetDiagField(i, SQL_DIAG_ROW_NUMBER, &nRowNumber, SQL_IS_INTEGER, NULL);

          USHORT nLineNumber = 0;
          GetDiagField(i, SQL_DIAG_SS_LINE, &nLineNumber, SQL_IS_INTEGER, NULL);

          SDWORD nMsgState = 0;
          GetDiagField(i, SQL_DIAG_SS_MSGSTATE, &nMsgState, SQL_IS_INTEGER, NULL);

          SDWORD nSeverity = 0;
          GetDiagField(i, SQL_DIAG_SS_SEVERITY, &nSeverity, SQL_IS_INTEGER, NULL);

          TCHAR szProcName[4096];
          szProcName[0] = _T('\0');
          SQLSMALLINT nProcName = 0;
          GetDiagField(i, SQL_DIAG_SS_PROCNAME, &szProcName, sizeof(szProcName), &nProcName);

        #ifdef SQL_MAX_SQLSEVERNAME
          TCHAR szServName[SQL_MAX_SQLSERVERNAME];
        #else
          TCHAR szServName[128];
        #endif
          szServName[0] = _T('\0');
          SQLSMALLINT nServName = 0;
          GetDiagField(i, SQL_DIAG_SS_SRVNAME, &szServName, sizeof(szServName), &nServName);    

          if (_tcslen(szProcName))
            ATLTRACE(_T(" Diag Record: State:%s, Native Error Code:%d, Message Text:%s, Row Number:%d, Message State:%d, Severity:%d, Proc Name:%s, Line Number:%d, Server Name:%s\n"), 
                     szState, nNativeError, szMessageText, nRowNumber, nMsgState, nSeverity, szProcName, nLineNumber, szServName);
          else
            ATLTRACE(_T(" Diag Record details: State:%s, Native Error Code:%d, Message Text:%s, Row Number:%d, Message State:%d, Severity:%d, Server Name:%s\n"), 
                     szState, nNativeError, szMessageText, nRowNumber, nMsgState, nSeverity, szServName);
        #else       
          ATLTRACE(_T(" Diag Record: State:%s, Native Error Code:%d, Message Text:%s\n"), szState, nNativeError, szMessageText);
        #endif //CODBCWRAPPERS_MSSQL_EXTENSIONS
        }
        else if (nGetDiagRet == SQL_NO_DATA)
          bContinue = FALSE;
      }
      
      ATLASSERT(FALSE);
    }
  #else
    (nRet);
  #endif
  }  

//Member variables
  SQLHANDLE m_h;
  SQLSMALLINT m_Type;
};


//Class which encapsulates an ODBC environment handle
class CEnvironment : public CHandle
{
public:
//Methods
  SQLRETURN Create()
  {
    //Delegate to the base class
    return CHandle::Create(SQL_HANDLE_ENV, SQL_NULL_HANDLE);
  }

  SQLRETURN DataSources(SQLUSMALLINT Direction, SQLTCHAR* ServerName, SQLSMALLINT BufferLength1, SQLSMALLINT* NameLength1Ptr,
                        SQLTCHAR* Description, SQLSMALLINT BufferLength2, SQLSMALLINT* NameLength2Ptr)
  {    
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLDataSources(m_h, Direction, ServerName, BufferLength1, NameLength1Ptr, Description, BufferLength2, NameLength2Ptr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN DataSources(SQLUSMALLINT Direction, String& sName, String& sDescription)
  {    
    SQLSMALLINT nNameLength = 0;
    SQLSMALLINT nDescLength = 0;
    SQLRETURN nRet = DataSources(Direction, NULL, 0, &nNameLength, NULL, 0, &nDescLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLTCHAR* pszTempName = reinterpret_cast<SQLTCHAR*>(sName.GetBufferSetLength(nNameLength + 1));
      SQLTCHAR* pszTempDesc = reinterpret_cast<SQLTCHAR*>(sDescription.GetBufferSetLength(nDescLength + 1));
      nRet = DataSources(Direction, pszTempName, static_cast<SQLSMALLINT>(nNameLength + 1), &nNameLength, pszTempDesc, static_cast<SQLSMALLINT>(nDescLength + 1), &nDescLength);
      sDescription.ReleaseBuffer();
      sName.ReleaseBuffer();
    }
    
    return nRet;
  }
#else
  SQLRETURN DataSources(SQLUSMALLINT Direction, String& sName, String& sDescription)
  {    
    SQLSMALLINT nNameLength = 0;
    SQLSMALLINT nDescLength = 0;
    SQLRETURN nRet = DataSources(Direction, NULL, 0, &nNameLength, NULL, 0, &nDescLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      std::vector<SQLTCHAR> sTempName;
      sTempName.resize(nNameLength + 1);
      std::vector<SQLTCHAR> sTempDesc;
      sTempDesc.resize(nDescLength + 1);
      nRet = DataSources(Direction, &(sTempName[0]), nNameLength + 1, &nNameLength, &(sTempDesc[0]), nDescLength + 1, &nDescLength);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
      {
        sName = reinterpret_cast<TCHAR*>(&(sTempName[0]));
        sDescription = reinterpret_cast<TCHAR*>(&(sTempDesc[0]));
      }
    }
    
    return nRet;
  }
#endif

  SQLRETURN Drivers(SQLUSMALLINT Direction, SQLTCHAR* DriverDescription, SQLSMALLINT BufferLength1, SQLSMALLINT* DescriptionLengthPtr,
                    SQLTCHAR* DriverAttributes, SQLSMALLINT BufferLength2, SQLSMALLINT* AttributesLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLDrivers(m_h, Direction, DriverDescription, BufferLength1, DescriptionLengthPtr, DriverAttributes, BufferLength2, AttributesLengthPtr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN Drivers(SQLUSMALLINT Direction, String& sDesc, CArray<String, String&>& attributes)
  {
    SQLSMALLINT nDescriptionLength = 0;
    SQLSMALLINT nAttributesLength = 0;
    SQLRETURN nRet = Drivers(Direction, NULL, 0, &nDescriptionLength, NULL, 0, &nAttributesLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLTCHAR* pszTempDesc = reinterpret_cast<SQLTCHAR*>(sDesc.GetBufferSetLength(nDescriptionLength + 1));
      CString sTempAttributes;
      SQLTCHAR* pszTempAttributes = reinterpret_cast<SQLTCHAR*>(sTempAttributes.GetBufferSetLength(nAttributesLength + 1));
      nRet = Drivers(Direction, pszTempDesc, static_cast<SQLSMALLINT>(nDescriptionLength + 1), &nDescriptionLength, pszTempAttributes, static_cast<SQLSMALLINT>(nAttributesLength + 1), &nAttributesLength);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        MultiSZToStringVector(pszTempAttributes, attributes);
      sTempAttributes.ReleaseBuffer();
      sDesc.ReleaseBuffer();
    }
    
    return nRet;
  }
#else
  SQLRETURN Drivers(SQLUSMALLINT Direction, String& sDesc, std::vector<String>& attributes)
  {
    SQLSMALLINT nDescriptionLength = 0;
    SQLSMALLINT nAttributesLength = 0;
    SQLRETURN nRet = Drivers(Direction, NULL, 0, &nDescriptionLength, NULL, 0, &nAttributesLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      std::vector<SQLTCHAR> sTempDesc;
      sTempDesc.resize(nDescriptionLength + 1);
      std::vector<SQLTCHAR> sTempAttributes;
      sTempAttributes.resize(nAttributesLength + 1);
      nRet = Drivers(Direction, &(sTempDesc[0]), nDescriptionLength + 1, &nDescriptionLength, &(sTempAttributes[0]), nAttributesLength + 1, &nAttributesLength);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
      {
        sDesc = reinterpret_cast<TCHAR*>(&(sTempDesc[0]));
        MultiSZToStringVector(&(sTempAttributes[0]), attributes);
      }
    }
    
    return nRet;
  }
#endif
  
  SQLRETURN EndTran(SQLSMALLINT CompletionType)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLEndTran(SQL_HANDLE_ENV, m_h, CompletionType);
  }

  SQLRETURN CommitTran()
  {
    return EndTran(SQL_COMMIT);
  }

  SQLRETURN RollbackTran()
  {
    return EndTran(SQL_ROLLBACK);
  }

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLGetEnvAttr(m_h, Attribute, ValuePtr, BufferLength, StringLengthPtr);
  }

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_INTEGER, NULL);
  }

  SQLRETURN GetAttrU(SQLINTEGER Attribute, SQLUINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_UINTEGER, NULL);
  }

  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER StringLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLSetEnvAttr(m_h, Attribute, ValuePtr, StringLength);
  }
  
  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_INTEGER);
  }

  SQLRETURN SetAttrU(SQLINTEGER Attribute, SQLUINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_UINTEGER);
  }
};


//Class which encapsulates an ODBC connection handle
class CConnection : public CHandle
{
public:
//Methods
  SQLRETURN Create(CEnvironment& environment)
  {
    //Delegate to the base class
    return CHandle::Create(SQL_HANDLE_DBC, environment);
  }

	void Attach(SQLHANDLE h)
  {
    CHandle::Attach(SQL_HANDLE_DBC, h);
  }

  SQLRETURN BrowseConnect(SQLTCHAR* InConnectionString, SQLSMALLINT StringLength1, SQLTCHAR* OutConnectionString, SQLSMALLINT BufferLength, SQLSMALLINT* StringLength2Ptr)
  {
    //Validate our parameters
  #ifdef ATLASSUME
    ATLASSUME(m_h != SQL_NULL_HANDLE);
  #else
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  #endif

    return SQLBrowseConnect(m_h, InConnectionString, StringLength1, OutConnectionString, BufferLength, StringLength2Ptr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN BrowseConnect(TCHAR* InConnectionString, String& sOutConnectionString)
  {
    SQLSMALLINT nStringLength = 0;
    SQLRETURN nRet = BrowseConnect(reinterpret_cast<SQLTCHAR*>(InConnectionString), SQL_NTS, NULL, 0, &nStringLength);
    if (nRet == SQL_NEED_DATA)
    {
      SQLTCHAR* pszOutConnectionString = reinterpret_cast<SQLTCHAR*>(sOutConnectionString.GetBufferSetLength(nStringLength + 1));
      nRet = BrowseConnect(reinterpret_cast<SQLTCHAR*>(InConnectionString), SQL_NTS, pszOutConnectionString, static_cast<SQLSMALLINT>(nStringLength + 1), &nStringLength);
      sOutConnectionString.ReleaseBuffer();
    }
  
    return nRet;
  }
#else
  SQLRETURN BrowseConnect(TCHAR* InConnectionString, String& sOutConnectionString)
  {
    SQLSMALLINT nStringLength = 0;
    SQLRETURN nRet = BrowseConnect(reinterpret_cast<SQLTCHAR*>(InConnectionString), SQL_NTS, NULL, 0, &nStringLength);
    if (nRet == SQL_NEED_DATA)
    {
      std::vector<SQLTCHAR> sTempOutConnectionString;
      sTempOutConnectionString.resize(nStringLength + 1);
      nRet = BrowseConnect(reinterpret_cast<SQLTCHAR*>(InConnectionString), SQL_NTS, &(sTempOutConnectionString[0]), nStringLength + 1, &nStringLength);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO || nRet == SQL_NEED_DATA)
        sOutConnectionString = reinterpret_cast<TCHAR*>(&(sTempOutConnectionString[0]));
    }
  
    return nRet;
  }
#endif

  SQLRETURN Connect(SQLTCHAR* ServerName, SQLSMALLINT NameLength1, SQLTCHAR* UserName, SQLSMALLINT NameLength2, SQLTCHAR* Authentication, SQLSMALLINT NameLength3)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLConnect(m_h, ServerName, NameLength1, UserName, NameLength2, Authentication, NameLength3);
  }

  SQLRETURN Connect(TCHAR* ServerName, TCHAR* UserName, TCHAR* Authentication)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return Connect(reinterpret_cast<SQLTCHAR*>(ServerName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(UserName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(Authentication), SQL_NTS);
  }
  
  SQLRETURN DriverConnect(SQLHWND WindowHandle, SQLTCHAR* InConnectionString, SQLSMALLINT StringLength1, SQLTCHAR* OutConnectionString, SQLSMALLINT BufferLength,
                          SQLSMALLINT* StringLength2Ptr, SQLUSMALLINT DriverCompletion)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLDriverConnect(m_h, WindowHandle, InConnectionString, StringLength1, OutConnectionString, BufferLength, StringLength2Ptr, DriverCompletion);
  }

  SQLRETURN DriverConnect(SQLTCHAR* InConnectionString, String& OutConnectionString, SQLHWND WindowHandle = NULL, SQLUSMALLINT DriverCompletion = SQL_DRIVER_NOPROMPT)
  {
    SQLTCHAR szOutConnection[4096];
    szOutConnection[0] = _T('\0');
    SQLSMALLINT nOutConnectionLength = 0;
    SQLRETURN nRet = DriverConnect(WindowHandle, InConnectionString, static_cast<SQLSMALLINT>(_tcslen(reinterpret_cast<TCHAR*>(InConnectionString))), szOutConnection, sizeof(szOutConnection)/sizeof(SQLTCHAR), &nOutConnectionLength, DriverCompletion);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
      OutConnectionString = reinterpret_cast<TCHAR*>(szOutConnection);
    return nRet;
  }
  
  SQLRETURN Disconnect()
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLDisconnect(m_h);
  }

  SQLRETURN EndTran(SQLSMALLINT CompletionType)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLEndTran(SQL_HANDLE_DBC, m_h, CompletionType);
  }

  SQLRETURN CommitTran()
  {
    return EndTran(SQL_COMMIT);
  }

  SQLRETURN RollbackTran()
  {
    return EndTran(SQL_ROLLBACK);
  }

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLGetConnectAttr(m_h, Attribute, ValuePtr, BufferLength, StringLengthPtr);
  }

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_INTEGER, NULL);
  }

  SQLRETURN GetAttrU(SQLINTEGER Attribute, SQLUINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_UINTEGER, NULL);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN GetAttr(SQLINTEGER Attribute, String& sValue)
  {
    SQLINTEGER nLength = 0;
    SQLRETURN nRet = GetAttr(Attribute, NULL, 0, &nLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLINTEGER nCharacters = nLength / sizeof(SQLTCHAR);
      SQLTCHAR* pszTempValue = reinterpret_cast<SQLTCHAR*>(sValue.GetBufferSetLength(nCharacters + 1));
      nRet = GetAttr(Attribute, pszTempValue, nLength + sizeof(SQLTCHAR), &nLength); 
      sValue.ReleaseBuffer();
    }
    return nRet;
  }
#else
  SQLRETURN GetAttr(SQLINTEGER Attribute, String& sValue)
  {
    SQLINTEGER nLength = 0;
    SQLRETURN nRet = GetAttr(Attribute, NULL, 0, &nLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLINTEGER nCharacters = nLength / sizeof(SQLTCHAR);
      std::vector<SQLTCHAR> sTempValue;
      sTempValue.resize(nCharacters + 1);
      nRet = GetAttr(Attribute, &(sTempValue[0]), nLength + sizeof(SQLTCHAR), &nLength); 
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        sValue = reinterpret_cast<TCHAR*>(&(sTempValue[0]));
    }
    return nRet;
  }
#endif

  SQLRETURN GetInfo(SQLUSMALLINT InfoType, SQLPOINTER InfoValuePtr, SQLSMALLINT BufferLength, SQLSMALLINT* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLGetInfo(m_h, InfoType, InfoValuePtr, BufferLength, StringLengthPtr);
  }

  SQLRETURN NativeSql(SQLTCHAR* InStatementText, SQLINTEGER TextLength1, SQLTCHAR* OutStatementText, SQLINTEGER BufferLength, SQLINTEGER* TextLength2Ptr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLNativeSql(m_h, InStatementText, TextLength1, OutStatementText, BufferLength, TextLength2Ptr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN NativeSql(TCHAR* InStatementText, String& sOutStatementText)
  {
    SQLINTEGER nOutStatementTextLength = 0;
    SQLRETURN nRet = NativeSql(reinterpret_cast<SQLTCHAR*>(InStatementText), SQL_NTS, NULL, 0, &nOutStatementTextLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLTCHAR* pszTempOutStatementText = reinterpret_cast<SQLTCHAR*>(sOutStatementText.GetBufferSetLength(nOutStatementTextLength + 1));
      nRet = NativeSql(reinterpret_cast<SQLTCHAR*>(InStatementText), SQL_NTS, pszTempOutStatementText, nOutStatementTextLength + 1, &nOutStatementTextLength);
      sOutStatementText.ReleaseBuffer();
    }
    
    return nRet;
  }
#else
  SQLRETURN NativeSql(TCHAR* InStatementText, String& sOutStatementText)
  {
    SQLINTEGER nOutStatementTextLength = 0;
    SQLRETURN nRet = NativeSql(reinterpret_cast<SQLTCHAR*>(InStatementText), SQL_NTS, NULL, 0, &nOutStatementTextLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      std::vector<SQLTCHAR> sTempOutStatementText;
      sTempOutStatementText.resize(nOutStatementTextLength + 1);
      nRet = NativeSql(reinterpret_cast<SQLTCHAR*>(InStatementText), SQL_NTS, &(sTempOutStatementText[0]), nOutStatementTextLength + 1, &nOutStatementTextLength);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        sOutStatementText = reinterpret_cast<TCHAR*>(&(sTempOutStatementText[0]));
    }
    
    return nRet;
  }
#endif

  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER StringLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLSetConnectAttr(m_h, Attribute, ValuePtr, StringLength);
  }

  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_INTEGER);
  }

  SQLRETURN SetAttrU(SQLINTEGER Attribute, SQLUINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_UINTEGER);
  }

  SQLRETURN SetAttr(SQLINTEGER Attribute, TCHAR* pszValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLTCHAR*>(pszValue), static_cast<SQLINTEGER>(_tcslen(reinterpret_cast<TCHAR*>(pszValue)) * sizeof(SQLTCHAR)));
  }
  
  SQLRETURN GetFunctions(SQLUSMALLINT FunctionId, SQLUSMALLINT* SupportedPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLGetFunctions(m_h, FunctionId, SupportedPtr);
  }
};


//Class which encapsulates an ODBC statement handle
class CStatement : public CHandle
{
public:
//Methods
  SQLRETURN Create(CConnection& connection)
  {
    //Delegate to the base class
    return CHandle::Create(SQL_HANDLE_STMT, connection);
  }

	void Attach(SQLHANDLE h)
  {
    CHandle::Attach(SQL_HANDLE_STMT, h);
  }

  SQLRETURN BindCol(SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValuePtr, SQLLEN BufferLength, SQLLEN* StrLen_or_Ind)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
    
    return SQLBindCol(m_h, ColumnNumber, TargetType, TargetValuePtr, BufferLength, StrLen_or_Ind);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, char& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, BYTE& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, long& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, bool& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, float& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, double& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, LONGLONG& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }
  
  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, BYTE* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, char* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, wchar_t* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, TIMESTAMP_STRUCT& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }
  
  SQLRETURN BindCol(SQLUSMALLINT ParameterNumber, SQL_NUMERIC_STRUCT& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindCol(ParameterNumber, _GetODBCCType(value), &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, SQLSMALLINT ValueType, SQLSMALLINT ParameterType, SQLULEN ColumnSize,
                          SQLSMALLINT DecimalDigits, SQLPOINTER ParameterValuePtr, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLBindParameter(m_h, ParameterNumber, InputOutputType, ValueType, ParameterType, ColumnSize,
                            DecimalDigits, ParameterValuePtr, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, char& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, BYTE& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, long& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, bool& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, float& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, double& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, LONGLONG& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }
  
  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, BYTE* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), BufferLength, 0, value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, char* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), BufferLength, 0, value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, wchar_t* value, SQLINTEGER BufferLength, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), BufferLength, 0, value, BufferLength, StrLen_or_IndPtr);
  }

  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, TIMESTAMP_STRUCT& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }
  
  SQLRETURN BindParameter(SQLUSMALLINT ParameterNumber, SQLSMALLINT InputOutputType, SQL_NUMERIC_STRUCT& value, SQLLEN* StrLen_or_IndPtr = NULL)
  {
    return BindParameter(ParameterNumber, InputOutputType, _GetODBCCType(value), _GetODBCSQLType(value), 0, 0, &value, 0, StrLen_or_IndPtr);
  }

  SQLRETURN BulkOperations(SQLUSMALLINT Operation)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLBulkOperations(m_h, Operation);
  }
  
  SQLRETURN Cancel()
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLCancel(m_h);
  }
  
  SQLRETURN CloseCursor()
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLCloseCursor(m_h);
  }
  
  SQLRETURN ColAttribute(SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier, SQLPOINTER CharacterAttributePtr, SQLSMALLINT BufferLength,
                         SQLSMALLINT* StringLengthPtr, SQLLEN* NumericAttributePtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLColAttribute(m_h, ColumnNumber, FieldIdentifier, CharacterAttributePtr, BufferLength, StringLengthPtr, NumericAttributePtr);
  }
  
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN ColAttribute(SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier, String& sValue, SQLLEN* NumericAttributePtr = NULL)
  {
    SQLSMALLINT nLength = 0;
    SQLRETURN nRet = ColAttribute(ColumnNumber, FieldIdentifier, NULL, 0, &nLength, NumericAttributePtr);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLINTEGER nCharacters = nLength / sizeof(SQLTCHAR);
      SQLTCHAR* pszTempValue = reinterpret_cast<SQLTCHAR*>(sValue.GetBufferSetLength(nCharacters + 1));
      nRet = ColAttribute(ColumnNumber, FieldIdentifier, pszTempValue, static_cast<SQLSMALLINT>(nLength + sizeof(SQLTCHAR)), &nLength, NumericAttributePtr); 
      sValue.ReleaseBuffer();
    }
    return nRet;
  }
#else
  SQLRETURN ColAttribute(SQLUSMALLINT ColumnNumber, SQLUSMALLINT FieldIdentifier, String& sValue, SQLLEN* NumericAttributePtr = NULL)
  {
    SQLSMALLINT nLength = 0;
    SQLRETURN nRet = ColAttribute(ColumnNumber, FieldIdentifier, NULL, 0, &nLength, NumericAttributePtr);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLINTEGER nCharacters = nLength / sizeof(SQLTCHAR);
      std::vector<SQLTCHAR> sTempValue;
      sTempValue.resize(nCharacters + 1);
      nRet = ColAttribute(ColumnNumber, FieldIdentifier, &(sTempValue[0]), nLength + sizeof(SQLTCHAR), &nLength, NumericAttributePtr); 
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        sValue = reinterpret_cast<TCHAR*>(&(sTempValue[0]));
    }
    return nRet;
  }
#endif
  
  SQLRETURN ColumnPrivileges(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName,
                             SQLSMALLINT NameLength3, SQLTCHAR* ColumnName, SQLSMALLINT NameLength4)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLColumnPrivileges(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3, ColumnName, NameLength4);
  }

  SQLRETURN ColumnPrivileges(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName, TCHAR* ColumnName)
  {
    return ColumnPrivileges(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(ColumnName), SQL_NTS);
  }

  SQLRETURN Columns(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName, 
                    SQLSMALLINT NameLength3, SQLTCHAR* ColumnName, SQLSMALLINT NameLength4)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLColumns(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3, ColumnName, NameLength4);
  }

  SQLRETURN Columns(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName, TCHAR* ColumnName)
  {
    return Columns(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(ColumnName), SQL_NTS);
  }
  
  SQLRETURN DescribeCol(SQLSMALLINT ColumnNumber, SQLTCHAR* ColumnName, SQLSMALLINT BufferLength, SQLSMALLINT* NameLengthPtr, 
                        SQLSMALLINT* DataTypePtr, SQLULEN* ColumnSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLDescribeCol(m_h, ColumnNumber, ColumnName, BufferLength, NameLengthPtr, DataTypePtr, ColumnSizePtr, DecimalDigitsPtr, NullablePtr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN DescribeCol(SQLSMALLINT ColumnNumber, String& sColumnName, SQLSMALLINT* DataTypePtr, SQLULEN* ColumnSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
  {
    SQLSMALLINT nNameLength = 0;
    SQLRETURN nRet = DescribeCol(ColumnNumber, NULL, 0, &nNameLength, DataTypePtr, ColumnSizePtr, DecimalDigitsPtr, NullablePtr);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLTCHAR* pszTempColumnName = reinterpret_cast<SQLTCHAR*>(sColumnName.GetBufferSetLength(nNameLength + 1));
      nRet = DescribeCol(ColumnNumber, pszTempColumnName, static_cast<SQLSMALLINT>(nNameLength + 1), &nNameLength, DataTypePtr, ColumnSizePtr, DecimalDigitsPtr, NullablePtr);
      sColumnName.ReleaseBuffer();
    }

    return nRet;
  }
#else
  SQLRETURN DescribeCol(SQLSMALLINT ColumnNumber, String& sColumnName, SQLSMALLINT* DataTypePtr, SQLULEN* ColumnSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
  {
    SQLSMALLINT nNameLength = 0;
    SQLRETURN nRet = DescribeCol(ColumnNumber, NULL, 0, &nNameLength, DataTypePtr, ColumnSizePtr, DecimalDigitsPtr, NullablePtr);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      std::vector<SQLTCHAR> sTempColumnName;
      sTempColumnName.resize(nNameLength + 1);
      nRet = DescribeCol(ColumnNumber, &(sTempColumnName[0]), nNameLength + 1, &nNameLength, DataTypePtr, ColumnSizePtr, DecimalDigitsPtr, NullablePtr);
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        sColumnName = reinterpret_cast<TCHAR*>(&(sTempColumnName[0]));
    }

    return nRet;
  }
#endif
  
  SQLRETURN DescribeParam(SQLUSMALLINT ParameterNumber, SQLSMALLINT* DataTypePtr, SQLULEN* ParameterSizePtr, SQLSMALLINT* DecimalDigitsPtr, SQLSMALLINT* NullablePtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLDescribeParam(m_h, ParameterNumber, DataTypePtr, ParameterSizePtr, DecimalDigitsPtr, NullablePtr);
  }

  SQLRETURN ExecDirect(SQLTCHAR* StatementText, SQLINTEGER TextLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLExecDirect(m_h, StatementText, TextLength);
  }

  SQLRETURN ExecDirect(TCHAR* StatementText)
  {
    return ExecDirect(reinterpret_cast<SQLTCHAR*>(StatementText), SQL_NTS);
  }
  
  SQLRETURN Execute()
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLExecute(m_h);
  }

  SQLRETURN Fetch()
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLFetch(m_h);
  }
  
  SQLRETURN FetchScroll(SQLSMALLINT FetchOrientation, SQLLEN FetchOffset)
  {  
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLFetchScroll(m_h, FetchOrientation, FetchOffset);
  }

  SQLRETURN FetchNext()
  {  
    return FetchScroll(SQL_FETCH_NEXT, 0);
  }

  SQLRETURN FetchPrior()
  {  
    return FetchScroll(SQL_FETCH_PRIOR, 0);
  }
  
  SQLRETURN FetchFirst()
  {  
    return FetchScroll(SQL_FETCH_FIRST, 0);
  }
  
  SQLRETURN FetchLast()
  {  
    return FetchScroll(SQL_FETCH_LAST, 0);
  }
 
  SQLRETURN FetchBookmark(SQLLEN FetchOffset)
  {  
    return FetchScroll(SQL_FETCH_BOOKMARK, FetchOffset);
  }

  SQLRETURN FetchAbsolute(SQLLEN FetchOffset)
  {  
    return FetchScroll(SQL_FETCH_ABSOLUTE, FetchOffset);
  }
  
  SQLRETURN FetchRelative(SQLLEN FetchOffset)
  {  
    return FetchScroll(SQL_FETCH_RELATIVE, FetchOffset);
  }

  SQLRETURN ForeignKeys(SQLTCHAR* PKCatalogName, SQLSMALLINT NameLength1, SQLTCHAR* PKSchemaName, SQLSMALLINT NameLength2, SQLTCHAR* PKTableName, SQLSMALLINT NameLength3, 
                        SQLTCHAR* FKCatalogName, SQLSMALLINT NameLength4, SQLTCHAR* FKSchemaName, SQLSMALLINT NameLength5, SQLTCHAR* FKTableName, SQLSMALLINT NameLength6)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLForeignKeys(m_h, PKCatalogName, NameLength1, PKSchemaName, NameLength2, PKTableName, NameLength3, FKCatalogName, NameLength4, FKSchemaName,
                          NameLength5, FKTableName, NameLength6);
  }

  SQLRETURN ForeignKeys(TCHAR* PKCatalogName, TCHAR* PKSchemaName, TCHAR* PKTableName, TCHAR* FKCatalogName, TCHAR* FKSchemaName, TCHAR* FKTableName)
  {
    return ForeignKeys(reinterpret_cast<SQLTCHAR*>(PKCatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(PKSchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(PKTableName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(FKCatalogName), 
                       SQL_NTS, reinterpret_cast<SQLTCHAR*>(FKSchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(FKTableName), SQL_NTS);
  }
  
  SQLRETURN Free(SQLUSMALLINT Option)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLFreeStmt(m_h, Option);
  }

  SQLRETURN GetCursorName(SQLTCHAR* CursorName, SQLSMALLINT BufferLength, SQLSMALLINT* NameLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLGetCursorName(m_h, CursorName, BufferLength, NameLengthPtr);
  }

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN GetCursorName(String& sCursorName)
  {
    SQLSMALLINT nLength = 0;
    SQLRETURN nRet = GetCursorName(NULL, 0, &nLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      SQLTCHAR* pszTempCursorName = reinterpret_cast<SQLTCHAR*>(sCursorName.GetBufferSetLength(nLength + 1));
      nRet = GetCursorName(pszTempCursorName, static_cast<SQLSMALLINT>(nLength + 1), &nLength);
      sCursorName.ReleaseBuffer();
    }
    return nRet;
  }
#else
  SQLRETURN GetCursorName(String& sCursorName)
  {
    SQLSMALLINT nLength = 0;
    SQLRETURN nRet = GetCursorName(NULL, 0, &nLength);
    if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
    {
      std::vector<SQLTCHAR> sTempCursorName;
      sTempCursorName.resize(nLength + 1);
      nRet = GetCursorName(&(sTempCursorName[0]), nLength + 1, &nLength); 
      if (nRet == SQL_SUCCESS || nRet == SQL_SUCCESS_WITH_INFO)
        sCursorName = reinterpret_cast<TCHAR*>(&(sTempCursorName[0]));
    }
    return nRet;
  }
#endif
  
  SQLRETURN GetData(SQLUSMALLINT ColumnNumber, SQLSMALLINT TargetType, SQLPOINTER TargetValuePtr, SQLLEN BufferLength, SQLLEN* StrLen_or_IndPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLGetData(m_h, ColumnNumber, TargetType, TargetValuePtr, BufferLength, StrLen_or_IndPtr);
  }  

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLGetStmtAttr(m_h, Attribute, ValuePtr, BufferLength, StringLengthPtr);
  }

  SQLRETURN GetAttr(SQLINTEGER Attribute, SQLINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_INTEGER, NULL);
  }

  SQLRETURN GetAttrU(SQLINTEGER Attribute, SQLUINTEGER& nValue)
  {
    return GetAttr(Attribute, &nValue, SQL_IS_UINTEGER, NULL);
  }

  SQLRETURN GetTypeInfo(SQLSMALLINT DataType)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLGetTypeInfo(m_h, DataType);
  }

  SQLRETURN MoreResults()
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLMoreResults(m_h);
  }
  
  SQLRETURN NumParams(SQLSMALLINT* ParameterCountPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLNumParams(m_h, ParameterCountPtr);
  }
  
  SQLRETURN NumResultCols(SQLSMALLINT* ColumnCountPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLNumResultCols(m_h, ColumnCountPtr);
  }
  
  SQLRETURN ParamData(SQLPOINTER* ValuePtrPtr)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLParamData(m_h, ValuePtrPtr);
  }

  SQLRETURN Prepare(SQLTCHAR* StatementText, SQLINTEGER TextLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLPrepare(m_h, StatementText, TextLength);
  }

  SQLRETURN Prepare(SQLTCHAR* StatementText)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLPrepare(m_h, StatementText, SQL_NTS);
  }

  SQLRETURN PrimaryKeys(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName, SQLSMALLINT NameLength3)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLPrimaryKeys(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3);
  }

  SQLRETURN PrimaryKeys(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName)
  {
    return PrimaryKeys(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS);
  }
  
  SQLRETURN ProcedureColumns(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* ProcName, SQLSMALLINT NameLength3, SQLTCHAR* ColumnName, SQLSMALLINT NameLength4)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);

    return SQLProcedureColumns(m_h, CatalogName, NameLength1, SchemaName, NameLength2, ProcName, NameLength3, ColumnName, NameLength4);
  }

  SQLRETURN ProcedureColumns(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* ProcName, TCHAR* ColumnName)
  {
    return ProcedureColumns(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(ProcName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(ColumnName), SQL_NTS);
  }
  
  SQLRETURN Procedures(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* ProcName, SQLSMALLINT NameLength3)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLProcedures(m_h, CatalogName, NameLength1, SchemaName, NameLength2, ProcName, NameLength3);
  }

  SQLRETURN Procedures(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* ProcName)
  {
    return Procedures(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(ProcName), SQL_NTS);
  }
  
  SQLRETURN PutData(SQLPOINTER DataPtr, SQLLEN StrLen_or_Ind)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLPutData(m_h, DataPtr, StrLen_or_Ind);
  }
  
  SQLRETURN RowCount(SQLLEN* RowCountPtr)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);
  
    return SQLRowCount(m_h, RowCountPtr);
  }
  
  SQLRETURN SetCursorName(SQLTCHAR* CursorName, SQLSMALLINT NameLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLSetCursorName(m_h, CursorName, NameLength);
  }

  SQLRETURN SetCursorName(SQLTCHAR* CursorName)
  {
    return SetCursorName(CursorName, SQL_NTS);
  }
  
  SQLRETURN SetPos(SQLSETPOSIROW RowNumber, SQLUSMALLINT Operation, SQLUSMALLINT LockType)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLSetPos(m_h, RowNumber, Operation, LockType);
  }
  
  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLPOINTER ValuePtr, SQLINTEGER StringLength)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLSetStmtAttr(m_h, Attribute, ValuePtr, StringLength);
  }
  
  SQLRETURN SetAttr(SQLINTEGER Attribute, SQLINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_INTEGER);
  }

  SQLRETURN SetAttrU(SQLINTEGER Attribute, SQLUINTEGER nValue)
  {
    return SetAttr(Attribute, reinterpret_cast<SQLPOINTER*>(nValue), SQL_IS_UINTEGER);
  }
  
  SQLRETURN SpecialColumns(SQLSMALLINT IdentifierType, SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2,
                           SQLTCHAR* TableName, SQLSMALLINT NameLength3, SQLSMALLINT Scope, SQLSMALLINT Nullable)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLSpecialColumns(m_h, IdentifierType, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3, Scope, Nullable);
  }

  SQLRETURN SpecialColumns(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName, SQLSMALLINT Scope, SQLSMALLINT Nullable)
  {
    return Statistics(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS, Scope, Nullable);
  }
  
  SQLRETURN Statistics(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName,
                       SQLSMALLINT NameLength3, SQLUSMALLINT Unique, SQLUSMALLINT Reserved)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLStatistics(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3, Unique, Reserved);
  }

  SQLRETURN Statistics(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName, SQLUSMALLINT Unique, SQLUSMALLINT Reserved)
  {
    return Statistics(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS, Unique, Reserved);
  }
  
  SQLRETURN TablePrivileges(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName, SQLSMALLINT NameLength3)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLTablePrivileges(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3);
  }

  SQLRETURN TablePrivileges(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName)
  {
    return TablePrivileges(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS);
  }
  
  SQLRETURN Tables(SQLTCHAR* CatalogName, SQLSMALLINT NameLength1, SQLTCHAR* SchemaName, SQLSMALLINT NameLength2, SQLTCHAR* TableName, SQLSMALLINT NameLength3,
                   SQLTCHAR* TableType, SQLSMALLINT NameLength4)
  {
    //Validate our parameters
    ATLASSERT(m_h != SQL_NULL_HANDLE);
  
    return SQLTables(m_h, CatalogName, NameLength1, SchemaName, NameLength2, TableName, NameLength3, TableType, NameLength4);
  }

  SQLRETURN Tables(TCHAR* CatalogName, TCHAR* SchemaName, TCHAR* TableName, TCHAR* TableType)
  {
    return Tables(reinterpret_cast<SQLTCHAR*>(CatalogName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(SchemaName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableName), SQL_NTS, reinterpret_cast<SQLTCHAR*>(TableType), SQL_NTS);
  }
};


//Class which encapsulates an ODBC descriptor handle
class CDescriptor : public CHandle
{
public:
//Methods
  SQLRETURN Create(CConnection& connection)
  {
    //Delegate to the base class
    return CHandle::Create(SQL_HANDLE_DESC, connection);
  }
  
	void Attach(SQLHANDLE h)
  {
    CHandle::Attach(SQL_HANDLE_DESC, h);
  }

  SQLRETURN Copy(SQLHDESC Target)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLCopyDesc(m_h, Target);
  }

  SQLRETURN GetField(SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier, SQLPOINTER ValuePtr, SQLINTEGER BufferLength, SQLINTEGER* StringLengthPtr)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLGetDescField(m_h, RecNumber, FieldIdentifier, ValuePtr, BufferLength, StringLengthPtr);
  }

  SQLRETURN GetRec(SQLSMALLINT RecNumber, SQLTCHAR* Name, SQLSMALLINT BufferLength, SQLSMALLINT* StringLengthPtr, SQLSMALLINT* TypePtr, SQLSMALLINT* SubTypePtr,
                   SQLLEN* LengthPtr, SQLSMALLINT* PrecisionPtr, SQLSMALLINT* ScalePtr, SQLSMALLINT* NullablePtr)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLGetDescRec(m_h, RecNumber, Name, BufferLength, StringLengthPtr, TypePtr, SubTypePtr, LengthPtr, PrecisionPtr, ScalePtr, NullablePtr);
  }
  
  SQLRETURN SetField(SQLSMALLINT RecNumber, SQLSMALLINT FieldIdentifier, SQLPOINTER ValuePtr, SQLINTEGER BufferLength)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);

    return SQLSetDescField(m_h, RecNumber, FieldIdentifier, ValuePtr, BufferLength);
  }
  
  SQLRETURN SetRec(SQLSMALLINT RecNumber, SQLSMALLINT Type, SQLSMALLINT SubType, SQLLEN Length, SQLSMALLINT Precision, SQLSMALLINT Scale, SQLPOINTER DataPtr,
                   SQLLEN* StringLengthPtr, SQLLEN* IndicatorPtr)
  {
    //Validate our parameters
    ATLASSUME(m_h != SQL_NULL_HANDLE);
  
    return SQLSetDescRec(m_h, RecNumber, Type, SubType, Length, Precision, Scale, DataPtr, StringLengthPtr, IndicatorPtr);
  }
};


//The base class which accessor classes derive from. T is the class that contains the data that will be accessed
template<class T>
class CAccessor : public T
{
public:
//Methods
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN BindColumns(CStatement& statement)
  {
    (statement); //To eliminate potential unreferenced variables warnings
  
    //What will be the return value from this method
    SQLRETURN nRet = SQL_SUCCESS;
  
  #if (_MSC_VER > 1200) //__if_exists only exists on versions of VS later than VS 6
    __if_exists(_GetBindColumnEntries)
  #endif
    {
      //Work out the number of bound columns we have
      int nColumns = 0;
      nRet = _GetBindColumnEntries(NULL, &nColumns, NULL);
      ODBC_CHECK_RETURN(nRet, statement);
      
      //Allocate the parameter indicators array
      m_ColumnIndicators.RemoveAll();
      m_ColumnIndicators.SetSize(nColumns);
  
      //Bind the columns
      nRet = _GetBindColumnEntries(&statement, NULL, &m_ColumnIndicators);
      ODBC_CHECK_RETURN(nRet, statement);
    }
    return nRet;
  }
#else
  SQLRETURN BindColumns(CStatement& statement)
  {
    (statement);
  
    //What will be the return value from this method
    SQLRETURN nRet = SQL_SUCCESS;
  
  #if (_MSC_VER > 1200) //__if_exists only exists on versions of VS later than VS 6
    __if_exists(_GetBindColumnEntries)
  #endif
    {
      //Work out the number of bound columns we have
      int nColumns = 0;
      nRet = _GetBindColumnEntries(NULL, &nColumns, NULL);
      ODBC_CHECK_RETURN(nRet, statement);
      
      //Allocate the parameter indicators array
      m_ColumnIndicators.clear();
      m_ColumnIndicators.insert(m_ColumnIndicators.end(), nColumns, 0);
  
      //Bind the columns
      nRet = _GetBindColumnEntries(&statement, NULL, &m_ColumnIndicators);
      ODBC_CHECK_RETURN(nRet, statement);
    }
    return nRet;
  }
#endif

#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN BindParameters(CStatement& statement)
  {
    (statement); //To eliminate potential unreferenced variables warnings
  
    //What will be the return value from this method
    SQLRETURN nRet = SQL_SUCCESS;

  #if (_MSC_VER > 1200) //__if_exists only exists on versions of VS later than VS 6
    __if_exists(_GetBindParametersEntries)
  #endif
    {
      //Work out the number of bound parameters we have
      int nColumns = 0;
      nRet = _GetBindParametersEntries(NULL, &nColumns, NULL);
      ODBC_CHECK_RETURN(nRet, statement);
    
      //Allocate the temporary array of indicators based on the parameter count
      m_ParameterIndicators.SetSize(nColumns, 0);
    
      //Bind the parameters
	    nRet = _GetBindParametersEntries(&statement, NULL, &m_ParameterIndicators);
      ODBC_CHECK_RETURN(nRet, statement);
    }
    
    return nRet;  
  }
#else
  SQLRETURN BindParameters(CStatement& statement)
  {
    (statement);
  
    //What will be the return value from this method
    SQLRETURN nRet = SQL_SUCCESS;

  #if (_MSC_VER > 1200) //__if_exists only exists on versions of VS later than VS 6
    __if_exists(_GetBindParametersEntries)
  #endif
    {
      //Work out the number of bound parameters we have
      int nColumns = 0;
      nRet = _GetBindParametersEntries(NULL, &nColumns, NULL);
      ODBC_CHECK_RETURN(nRet, statement);
    
      //Allocate the temporary array of indicators based on the parameter count
      m_ParameterIndicators.clear();
      m_ParameterIndicators.insert(m_ParameterIndicators.end(), nColumns, 0);
    
      //Bind the parameters
	    nRet = _GetBindParametersEntries(&statement, NULL, &m_ParameterIndicators);
      ODBC_CHECK_RETURN(nRet, statement);
    }
    
    return nRet;  
  }
#endif
    	
//Member variables
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  CArray<SQLLEN, SQLLEN> m_ColumnIndicators;
  CArray<SQLLEN, SQLLEN> m_ParameterIndicators;
#else
  std::vector<SQLLEN> m_ColumnIndicators;
  std::vector<SQLLEN> m_ParameterIndicators;
#endif
};


//An accessor where the column data is bound dynamically
template<class T>
class CDynamicColumnAccessor : public CAccessor<T>
{
public:
//Constructors / Destructors
	CDynamicColumnAccessor() : m_nColumns(0)
	{
	}

//Methods
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  SQLRETURN BindColumns(CStatement& statement)
  {
    //Work out how many columns there are in the recordset
    m_nColumns = 0;
    SQLRETURN nRet = statement.NumResultCols(&m_nColumns);
    ODBC_CHECK_RETURN(nRet, statement);

    //Allocate the column indicators array
    m_ColumnIndicators.SetSize(m_nColumns); 
    
    //Also get the column details
    m_ColumnNames.SetSize(0, m_nColumns);
    m_DataTypes.SetSize(0, m_nColumns);
    m_ColumnSizes.SetSize(0, m_nColumns);
    m_DecimalDigits.SetSize(0, m_nColumns);
    m_Nullables.SetSize(0, m_nColumns);
    for (SQLSMALLINT i=1; i<=m_nColumns; i++)
    {
      String sColumnName;
      SQLSMALLINT nDataType = 0;
      SQLULEN nColumnSize = 0;
      SQLSMALLINT nDecimalDigits = 0;
      SQLSMALLINT nNullable = 0;
      nRet = statement.DescribeCol(i, sColumnName, &nDataType, &nColumnSize, &nDecimalDigits, &nNullable);
      ODBC_CHECK_RETURN(nRet, statement);
      m_ColumnNames.Add(sColumnName);
      m_DataTypes.Add(nDataType);
      m_ColumnSizes.Add(nColumnSize);
      m_DecimalDigits.Add(nDecimalDigits);
      m_Nullables.Add(nNullable);
    }
    
    ASSERT(static_cast<SQLSMALLINT>(m_ColumnIndicators.GetSize()) == m_nColumns);
    ASSERT(static_cast<SQLSMALLINT>(m_ColumnNames.GetSize()) == m_nColumns);
    ASSERT(static_cast<SQLSMALLINT>(m_DataTypes.GetSize()) == m_nColumns);
    ASSERT(static_cast<SQLSMALLINT>(m_ColumnSizes.GetSize()) == m_nColumns);
    ASSERT(static_cast<SQLSMALLINT>(m_DecimalDigits.GetSize()) == m_nColumns);
    ASSERT(static_cast<SQLSMALLINT>(m_Nullables.GetSize()) == m_nColumns);

    return nRet; 
  }
#else
  SQLRETURN BindColumns(CStatement& statement)
  {
    //Work out how many columns there are in the recordset
    m_nColumns = 0;
    SQLRETURN nRet = statement.NumResultCols(&m_nColumns);
    ODBC_CHECK_RETURN(nRet, statement);

    //Allocate the column indicators array
    m_ColumnIndicators.clear();
    m_ColumnIndicators.insert(m_ColumnIndicators.end(), m_nColumns, 0); 
    
    //Also get the column details
    m_ColumnNames.reserve(m_nColumns);
    m_DataTypes.reserve(m_nColumns);
    m_ColumnSizes.reserve(m_nColumns);
    m_DecimalDigits.reserve(m_nColumns);
    m_Nullables.reserve(m_nColumns);
    for (SQLSMALLINT i=1; i<=m_nColumns; i++)
    {
      String sColumnName;
      SQLSMALLINT nDataType = 0;
      SQLULEN nColumnSize = 0;
      SQLSMALLINT nDecimalDigits = 0;
      SQLSMALLINT nNullable = 0;
      nRet = statement.DescribeCol(i, sColumnName, &nDataType, &nColumnSize, &nDecimalDigits, &nNullable);
      ODBC_CHECK_RETURN(nRet, statement);
      m_ColumnNames.push_back(sColumnName);
      m_DataTypes.push_back(nDataType);
      m_ColumnSizes.push_back(nColumnSize);
      m_DecimalDigits.push_back(nDecimalDigits);
      m_Nullables.push_back(nNullable);
    }
    
    ATLASSERT(static_cast<SQLSMALLINT>(m_ColumnIndicators.size()) == m_nColumns);
    ATLASSERT(static_cast<SQLSMALLINT>(m_ColumnNames.size()) == m_nColumns);
    ATLASSERT(static_cast<SQLSMALLINT>(m_DataTypes.size()) == m_nColumns);
    ATLASSERT(static_cast<SQLSMALLINT>(m_ColumnSizes.size()) == m_nColumns);
    ATLASSERT(static_cast<SQLSMALLINT>(m_DecimalDigits.size()) == m_nColumns);
    ATLASSERT(static_cast<SQLSMALLINT>(m_Nullables.size()) == m_nColumns);

    return nRet; 
  }
#endif
  
  String GetColumnName(SQLSMALLINT nColumn) const
  {
    return m_ColumnNames[nColumn - 1]; //nColumn is 1 based
  }
 
  SQLSMALLINT GetColumnType(SQLSMALLINT nColumn) const
  {
    return m_ColumnDataTypes[nColumn - 1]; //nColumn is 1 based
  }

  SQLULEN GetColumnSize(SQLSMALLINT nColumn) const
  {
    return m_ColumnSizes[nColumn - 1]; //nColumn is 1 based
  }

  SQLSMALLINT GetColumnDecimalDigits(SQLSMALLINT nColumn) const
  {
    return m_ColumnDecimalDigits[nColumn - 1]; //nColumn is 1 based
  }
  
  SQLSMALLINT GetColumnNullables(SQLSMALLINT nColumn) const
  {
    return m_ColumnNullables[nColumn - 1]; //nColumn is 1 based
  }
  
	SQLSMALLINT GetColumnNo(LPCTSTR pszColumnName) const
	{
		//What will be the return value from this function
		SQLSMALLINT nColumn = 0;
		
		for (SQLSMALLINT i=0; (i<m_nColumns) && (nColumn == 0); i++)
		{
		  if (m_ColumnNames[i] == pszColumnName)
		    nColumn = static_cast<SQLSMALLINT>(i + 1); //Columns are one based
		}
		
		return nColumn;
	}

	template <class ctype>
	SQLRETURN GetValue(CStatement& statement, SQLSMALLINT nColumn, ctype& data, SQLLEN* StrLen_or_IndPtr = NULL)
	{
    SQLLEN* pIndicator = StrLen_or_IndPtr;
    if (pIndicator == NULL)
    {
      SQLLEN& nColumnIndicator = m_ColumnIndicators[nColumn - 1];
      pIndicator = &nColumnIndicator;
    }
	  return statement.GetData(nColumn, _GetODBCCType(data), &data, sizeof(ctype), pIndicator);
	}
  
//Member variables
  SQLSMALLINT                       m_nColumns;
#ifdef CODBCWRAPPERS_MFC_EXTENSIONS
  CArray<String, String&>           m_ColumnNames;
  CArray<SQLSMALLINT, SQLSMALLINT>  m_DataTypes;
  CArray<SQLULEN, SQLULEN>          m_ColumnSizes;
  CArray<SQLSMALLINT, SQLSMALLINT>  m_DecimalDigits;
  CArray<SQLSMALLINT, SQLSMALLINT>  m_Nullables;
#else
  std::vector<String>               m_ColumnNames;
  std::vector<SQLSMALLINT>          m_DataTypes;
  std::vector<SQLULEN>              m_ColumnSizes;
  std::vector<SQLSMALLINT>          m_DecimalDigits;
  std::vector<SQLSMALLINT>          m_Nullables;
#endif
};
  
  
}; //namespace COLEDB


#endif //__ODBCWRAPPERS_H__